#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import os
import logging
from time import time

from terra.core.manager import Manager
from terra.core.model import ModelFolder, Model
from terra.core.plugin_prefs import PluginPrefs
from terra.core.singleton import Singleton
from terra.core.task import Task
from terra.utils.encoding import to_utf8

from playlist_model import ListPlaylistMedia

__all__ = ("AudioLocalModel", "AudioLocalModelFolder",
           "AudioLocalAllModelFolder", "AudioLocalRandomModelFolder",
           "AudioLocalPlaylistModelFolder", "AudioLocalArtistModelFolder",
           "AudioLocalPlaylistMediaModelFolder", "AudioOnTheGoModelFolder",
           "AudioPlaylistOTGHook", "AudioResetDBHook",
           "AudioLocalArtistMediaModelFolder",
           "AudioLocalArtistMenuModelFolder", "AudioLocalAlbumMediaModelFolder",
           "AudioLocalAlbumModelFolder", "AlbumArtAudioModelFolder",
           "AudioLocalFavouritesModelFolder", "AudioLocalRatingModelFolder",
           "AudioLocalGenreModelFolder", "AudioLocalGenreMediaModelFolder",
           "MyMusicSubMenuModelFolder", "MyMusicIcon")


mger = Manager()
AudioModel = mger.get_class("Model/Media/Audio")
LazyDBList = mger.get_class("Utils/LazyDBList")
HybridList = mger.get_class("Utils/HybridList")
ModelFolderOrdered = mger.get_class("Model/Folder/Ordered")
DefaultIcon = mger.get_class("Icon")
OnTheGoHook = mger.get_class("Hook/Abstract/OnTheGo")
Notify = mger.get_class("Model/Notify")
AudioDBDirectoryModel = mger.get_class("Model/Folder/Directory/Media/Audio")

db = mger.canola_db

log = logging.getLogger("canola.audio")


class MyMusicIcon(DefaultIcon):
    terra_type = "Icon/Folder/Task/Audio/Local"
    icon = "icon/main_item/music_local"


class Query(object):
    def __init__(self):
        self._id = None
        self._stmt_tmpl = "SELECT %%s FROM %s %%s" % self.table

    def __set_id(self, value):
        self._id = value

    def __get_id(self):
        raise NotImplementedError("You must implement __get_id")

    def do_query(self, fields, clause=None, *args):
        if clause is None:
            clause = ""
        else:
            clause = "WHERE %s" % clause
        stmt = self._stmt_tmpl % (fields, clause)
        return db.execute(stmt, *args)

    def do_query_first(self, fields, clause=None, *args):
        try:
            return self.do_query(fields, clause, *args)[0]
        except IndexError:
            return None

    id = property(__get_id, __set_id)

class Artist(Query):
    table = "audio_artists"
    clause_get_id = "name = ?"
    clause_get_name = "id = ?"
    stmt_get_albums = "SELECT id, name FROM audio_albums "\
                      "WHERE artist_id = ? AND name != ''"

    def __init__(self):
        Query.__init__(self)
        self.__name = None
        self.__albums = None

    def __get_id(self):
        """Get Artist's id based on it's name.

        If name is 'None' or there is not an artist with that
        name returns 'None'.
        """
        if not self._id and self.__name:
            row = self.do_query_first("id", self.clause_get_id, (self.__name,))
            self._id = row[0]
        return self._id

    def __set_name(self, value):
        self.__name = value

    def __get_name(self):
        """Get Artist's name based on it's id.

        If id is 'None' or there is not an artist with that
        id returns 'None'.
        """
        if self._id and not self.__name:
            row = self.do_query_first("name", self.clause_get_name, (self._id,))
            self.__name = row[0].encode('utf-8')
        return self.__name

    def __get_albums(self):
        """Get all albums related to this artist.

        """
        if self._id and not self.__albums:
            rows = db.execute(self.stmt_get_albums, (self._id,))
            self.__albums = [(i, n.encode("utf-8")) for i, n in rows]
        return self.__albums

    name = property(__get_name, __set_name)
    albums = property(__get_albums)


class Album(Query):
    table = "audio_albums"
    clause_get_id = "name = ? AND artist_id = ?"
    clause_get_by_id = "id = ?"

    def __init__(self):
        Query.__init__(self)
        self._name = None
        self._artist_id = None
        self._cover = None
        self._cover_small = None
        self.__artist = None
        self.titles = []

    def __get_id(self):
        """Get Albums's id based on it's name and artist's id.

        If artist's id or it's name is 'None' or there is not
        a corresponding album for that (artist_id,name) returns 'None'.
        """
        if not self._id and self._name and self._artist_id:
            self._id = self.do_query_first("id", self.clause_get_id,
                                           (self._name, self._artist_id))[0]
        return self._id

    def _get_all_by_id(self):
        """Get all Album info based on it's id."""
        if self._id and not self._name and not self._artist_id:
            row = self.do_query_first("name, artist_id",
                                      self.clause_get_by_id, (self._id,))
            self._name = row[0].encode('utf-8')
            self._artist_id = row[1]

    def __get_name(self):
        if not self._name and self._id:
            self._get_all_by_id()
        return self._name

    def __get_artist_id(self):
        if not self._artist_id and self._id:
            self._get_all_by_id()
        return self._artist_id

    def __get_artist(self):
        if not self.__artist:
            artist = Artist()
            artist.id = self.artist_id
            self.__artist = artist.name
        return self.__artist

    def _load_cover_path(self):
        canola_prefs = PluginPrefs("settings")
        try:
            path = canola_prefs["cover_path"]
        except KeyError:
            path = canola_prefs["cover_path"] = \
                os.path.expanduser("~/.canola/covers/")
            canola_prefs.save()

        if not os.path.exists(path):
            os.makedirs(path)
        return path

    def __get_cover(self):
        if not self._cover and self._id:
            path = self._load_cover_path()
            artist = self.artist
            name = self.name
            try:
                self._cover = os.path.join(path, artist, name, "cover.jpg")
            except Exception:
                self._cover = None
        return self._cover

    def __get_cover_small(self):
        if not self._cover_small and self._id:
            path = self._load_cover_path()
            artist = self.artist
            name = self.name
            cover = "%s/%s/cover-small.jpg" % (artist, name)
            self._cover_small = os.path.join(path, cover)
        return self._cover_small

    name = property(__get_name)
    artist_id = property(__get_artist_id)
    cover = property(__get_cover)
    cover_small = property(__get_cover_small)
    artist = property(__get_artist)


class AudioLocalModel(AudioModel):
    terra_type = "Model/Media/Audio/Local"
    table = "audios"
    stmt_get_uri = "SELECT path FROM files WHERE id = ?"
    stmt_get_genre = "SELECT name FROM audio_genres WHERE id = ?"
    stmt_update_rating = "UPDATE audios SET rating = ? WHERE id = ?"
    stmt_update_playcnt = "UPDATE audios SET playcnt = ? WHERE id = ?"

    __slots__ = ("cur", "parent", "id", "title", "genre_id", "_rating",
                 "artist_id", "album_id", "__uri", "__genre", "__artist",
                 "__album", "__cover")

    def __init__(self, parent=None):
        AudioModel.__init__(self)
        self.parent = parent
        self.id = None
        self.title = None
        self.genre_id = None
        self.album_id = None
        self._rating = None
        self._playcnt = 0
        self.uri = None
        self.__genre = None
        self.__artist = None
        self.__album = None
        self.__cover = None

    def __get_genre(self):
        if self.genre_id and not self.__genre:
            row = db.execute(self.stmt_get_genre, (self.genre_id,))[0]
            self.__genre = row[0].encode('utf-8')
        return self.__genre

    def __set_rating(self, value):
        if value is None or self.id is None:
            self._rating = None
        else:
            db.execute(self.stmt_update_rating, (value, self.id))
            self._rating = value

    def __get_rating(self):
        return self._rating

    def __set_playcnt(self, value):
        if value and self.id:
            db.execute(self.stmt_update_playcnt, (value, self.id))
            self._playcnt = value

    def __get_playcnt(self):
        return self._playcnt

    def __get_artist(self):
        if not self.__artist:
            aux_album = Album()
            aux_album.id = self.album_id
            aux = Artist()
            aux.id = aux_album.artist_id
            self.__artist = aux.name
        return self.__artist

    def __get_album(self):
        if not self.__album:
            aux = Album()
            aux.id = self.album_id
            self.__album = aux.name
        return self.__album

    def __get_cover(self):
        if not self.__cover:
            aux = Album()
            aux.id = self.album_id
            self.__cover = aux.cover
        return self.__cover

    def __dummy_setter(self, value):
        # dummy setter to stay compatible
        return None

    def __get_name(self):
        return self.title

    name = property(__get_name, __dummy_setter)
    cover = property(__get_cover, __dummy_setter)
    genre = property(__get_genre, __dummy_setter)
    album = property(__get_album, __dummy_setter)
    artist = property(__get_artist, __dummy_setter)
    rating = property(__get_rating, __set_rating)
    playcnt = property(__get_playcnt, __set_playcnt)


class ListLocalMedia(LazyDBList):
    def _create_item(self, row):
        item = AudioLocalModel(self.parent)
        item.id = row[0]
        item.title = row[1].encode('utf-8')
        item.album_id = row[2]
        item.artist_id = row[3]
        item.genre_id = row[4]
        item.trackno = row[5]
        item._rating = row[6]
        item._playcnt = row[7]
        # need this because it's BLOB
        item.uri = str(row[8])
        return item

class ListLocalAlbums(LazyDBList):
    def _create_item(self, row):
        item = AudioLocalAlbumMediaModelFolder((row[0], row[1]))
        item.parent = self.parent
        return item


class ListLocalArtists(LazyDBList):
    def _create_item(self, row):
        item = AudioLocalArtistMenuModelFolder((row[0], row[1]))
        item.parent = self.parent
        return item


class ListLocalPlaylists(LazyDBList):
    "LazyDBList that contains the playlists entries found in the DB"
    delete_stmt = "DELETE FROM playlist_ordered WHERE id = ?"

    def _create_item(self, row):
        item = AudioLocalPlaylistMediaModelFolder((row[0], row[1],
                                                   row[2], row[3]))
        item.parent = self.parent
        return item

    def _on_item_delete(self, item):
        "Update temp table"
        db.execute(self.delete_stmt, (item.id,))

    def remove(self, model):
        del self[self._elements.index(model)]


class HybridPlaylistsList(HybridList):
    "HybridList that adds a static element on top of ListLocalPlaylists"
    def __init__(self, parent, query, *query_args):
        self.parent = parent
        static = self._init_static_list()
        lazydb = ListLocalPlaylists(parent, query, *query_args)
        HybridList.__init__(self, static, lazydb)

    def _init_static_list(self):
        # Add the OTG menu item. One can move this around if desired.
        # OTG is a singleton thus it's not possible to set its parent and
        # name using its constructor. You must get the instance and then set it.
        otg_instance = AudioOnTheGoModelFolder()
        otg_instance.parent = self.parent
        otg_instance.name = "On The Move"

        return [otg_instance]

    def reset(self):
        self._second.reset()

    def is_static(self, model):
        return model in self._first

    def remove(self, model):
        self._second.remove(model)


class ListLocalPlaylistMedia(ListPlaylistMedia):
    def _create_item(self, row):
        item = AudioLocalModel(self.parent)
        item.id = row[0]
        item.title = row[1].encode('utf-8')
        item.album_id = row[2]
        item.artist_id = row[3]
        item.genre_id = row[4]
        item.trackno = row[5]
        item._rating = row[6]
        item._playcnt = row[7]
        # need this because it's BLOB
        item.uri = str(row[8])
        return item


class AudioLocalModelFolder(ModelFolder):
    clear_stmt = "DROP TABLE IF EXISTS %s"
    has_onthego = True
    children_list_cls = ListLocalMedia
    table = ""

    def __init__(self, name, parent=None):
        ModelFolder.__init__(self, name, parent, None)
        self._create_children_list()
        self._tables_created = False

    def _create_children_list(self):
        self.children = self.children_list_cls(self, self.table)

    def commit_changes(self):
        mger.canola_db.commit()
        log.info("Committed info to DB")

    def _setup_tmp_table(self):
        cur = db.get_cursor()
        try:
            cur.execute(self.clear_stmt % self.table)
        except Exception, e:
            #sqlite may raise a "schema has changed" error after the user has
            #just rescanned his collection, but it's no harm.
            log.error("%s db error: %s", self.__class__, e)

        cur.execute(self.create_stmt % self.table)
        self.commit_changes()
        self._tables_created = True

    def do_load(self):
        try:
            self._setup_tmp_table()
            self.children.reset()
        except Exception, e:
            log.error("%s load error: %s", self.__class__, e)

    def do_unload(self):
        self.children.reset()

    def __str__(self):
        if self._tables_created:
            c = bool(self.children)
        else:
            c = False
        return "%s(name=%r, parent=%s, children=%s)" % \
               (self.__class__.__name__, self.name, bool(self.parent), c)

    __repr__ = __str__

class AudioLocalAllModelFolder(AudioLocalModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/All"
    table = "all_ordered"
    has_onthego = True
    create_stmt = "CREATE TEMP TABLE %s AS " \
                  "SELECT * FROM audios,files USING (id) WHERE " \
                  "files.dtime = 0 ORDER BY audios.title COLLATE NOCASE"


class AudioLocalRandomModelFolder(AudioLocalModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/Random"
    table = "audios_random"
    has_onthego = True
    create_stmt = "CREATE TEMP TABLE %s AS " \
                  "SELECT * FROM audios,files USING (id) WHERE " \
                  "files.dtime = 0 ORDER BY RANDOM(*)"


class AudioLocalPlaylistMediaModelFolder(AudioLocalModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/Playlist/Songs"
    has_onthego = True

    def __init__(self, playlist, parent=None):
        self.id = playlist[0]
        self.name = to_utf8(playlist[1])
        self.n_items = playlist[2]
        self.path = to_utf8(str(playlist[3])) # because it's a buffer
        AudioLocalModelFolder.__init__(self, self.name, parent)

    def _setup_tmp_table(self):
        pass # ListLocalPlaylistMedia don't need a tmp_table

    def _create_children_list(self):
        self.children = ListLocalPlaylistMedia(self, self.path, self.n_items)


class AudioLocalPlaylistModelFolder(AudioLocalModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/Playlists"
    children_list_cls = HybridPlaylistsList
    table = "playlist_ordered"
    delete_file_stmt = "UPDATE files SET dtime = ? WHERE id = ?"
    clear_stmt = "DROP TABLE IF EXISTS %s"
    create_stmt = "CREATE TEMP TABLE %s AS " \
                  "SELECT * FROM playlists,files USING (id) WHERE " \
                  "files.dtime = 0 ORDER BY playlists.title " \
                  "COLLATE NOCASE"

    def __init__(self, name, parent=None):
        AudioLocalModelFolder.__init__(self, name, parent)
        self._child_removed_callbacks = []

    def _notify_child_removed(self, model):
        for meth in self._child_removed_callbacks:
            meth(model)

    def add_child_removed_callback(self, cb):
        self._child_removed_callbacks.append(cb)

    def del_child_removed_callback(self, cb):
        self._child_removed_callbacks.remove(cb)

    def is_removable_child(self, model):
        return not self.children.is_static(model)

    def on_remove_child(self, model):
        try:
            os.remove(model.path)
        except OSError:
            pass
        db.execute(self.delete_file_stmt, (int(time()), model.id))
        self.children.remove(model)
        self.notify_model_changed()
        self._notify_child_removed(model)

    def update(self):
        self._setup_tmp_table()
        self.children.reset()
        self.notify_model_changed()

    def do_unload(self):
        self._child_removed_callbacks = []
        AudioLocalModelFolder.do_unload(self)


class AudioPlaylistOTGHook(OnTheGoHook):
    terra_type = "Hook/OnTheGo/AudioPlaylist"

    def __init__(self, controller):
        OnTheGoHook.__init__(self)
        self._controller = controller

    def playlist_saved(self, playlist_id):
        curr_song = self._controller.parent.get_playing_model()

        if curr_song and \
           isinstance(curr_song.parent, AudioLocalPlaylistMediaModelFolder) and \
           curr_song.parent.id == playlist_id:
            # A song from the overwritten playlist was playing. Kill the player.
            self._controller.parent.kill_player()
        # Refresh our model.
        self._controller.model.update()


class AudioResetDBHook(Model):
    terra_type = "Hook/ResetDB"

    def __init__(self, name=None, parent=None):
        Model.__init__(self, name, parent)

    def reset_db(self, con):
        cmds = ("DELETE FROM files WHERE id IN (SELECT files.id"
                " FROM files,audios WHERE files.id = audios.id)",
                "DELETE FROM audio_albums",
                "DELETE FROM audio_artists",
                "DELETE FROM audio_genres")

        for c in cmds:
            try:
                log.debug("executing %s over %s", c, con)
                con.execute(c)
                con.commit()
            except Exception, e:
                log.error("%s db error: %r %s", self.__class__, c, e)


class AudioOnTheGoModelFolder(Singleton, ModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/Playlist/OnTheGo"
    task_terra_type = "Task/Audio/LocalOnTheGo"
    query_file_stmt = "SELECT id FROM files WHERE path=?"
    update_file_stmt = "UPDATE files SET mtime=?, dtime=0, size=? WHERE id=?"
    add_file_stmt = "INSERT INTO files (path, mtime, dtime, size)" \
                    " VALUES(?, ?, ?, ?)"
    add_playlist_stmt = "INSERT OR REPLACE INTO playlists (id, " \
                        "title, n_entries) VALUES (?, ?, ?)"
    child_class = AudioLocalModel
    child_attrs = ("id", "title", "genre_id", "album_id", "uri", "trackno")

    def __init__(self, name="", parent=None, children=None):
        self._parent = None
        self._child_removed_callbacks = []
        Singleton.__init__(self)
        ModelFolder.__init__(self, name, parent, children)

    def _notify_child_removed(self, model):
        for meth in self._child_removed_callbacks:
            meth(model)

    def add_child_removed_callback(self, cb):
        self._child_removed_callbacks.append(cb)

    def del_child_removed_callback(self, cb):
        self._child_removed_callbacks.remove(cb)

    def is_removable_child(self, model):
        return True

    def on_remove_child(self, model):
        self.children.remove(model)
        self._notify_child_removed(model)

    def enqueue(self, item):
        self.children.append(self._create_item(item, self))

    def _create_item(self, item, parent=None):
        parent = parent or item.parent
        try:
            new_item = self.child_class(parent)
            for attr in self.child_attrs:
                setattr(new_item, attr, getattr(item, attr))
            return new_item
        except AttributeError, e:
            log.debug_error("Cannot create '%s' instance from '%s'.\n%s" %
                            (self.child_class, item.__class__, e))

    def add_to_db(self, filename):
        st = os.stat(filename)

        cur = mger.canola_db.get_cursor()
        r = cur.execute(self.query_file_stmt,
                             (buffer(to_utf8(filename)), )).fetchall()
        if r:
            primary_key = r[0][0]
            query_arguments = (st.st_mtime, st.st_size, primary_key)
            cur.execute(self.update_file_stmt, query_arguments)
            log.debug("updated previous entry id '%d'" % primary_key)
        else:
            cur.execute(self.add_file_stmt, (buffer(to_utf8(filename)),
                                                  st.st_mtime, 0, st.st_size))
            primary_key = cur.lastrowid
            log.debug("added new entry id '%d'" % primary_key)

        cur.execute(self.add_playlist_stmt,
                         (primary_key, to_utf8(os.path.basename(filename)[:-4]),
                          len(self.children)))

        mger.canola_db.commit()
        log.info("Commited info to DB")
        return primary_key

    def save_playlist(self, filename, dirname=None, force=False):
        if not isinstance(filename, str):
            return None
        filename = os.path.basename(filename)
        if not filename:
            return None

        if dirname is None or not isinstance(dirname, str):
            dirname = PluginPrefs("settings")["audio"][0]

        if not filename.endswith(".pls"):
            filename += ".pls"
        fullname = os.path.join(dirname, filename)

        if not os.path.exists(dirname):
            os.makedirs(dirname)

        if not force and os.path.exists(fullname):
            return None

        try:
            fo = file(fullname, "w")
        except IOError:
            return None

        fo.write("[playlist]\n")
        fo.write("NumberOfEntries=%d\n\n" % len(self.children))
        i = 1
        for item in self.children:
            fo.write("File%d=%s\n" % (i, item.uri))
            fo.write("Title%d=%s\n" % (i, item.title))
            fo.write("Length%d=-1\n\n" % i)
            i += 1
        fo.write("Version=2\n")
        fo.close()

        primary_key = self.add_to_db(fullname)

        return (primary_key, fullname)

    def clear_playlist(self):
        del self.children[:]

    def __get_parent(self):
        return self._parent

    def __set_parent(self, parent):
        self._parent = parent
        if parent is not None and self not in parent.children:
            parent.append(self)

    parent = property(__get_parent, __set_parent)

    def do_unload(self):
        self._child_removed_callbacks = []
        ModelFolder.do_unload(self)
        self._parent = None


class AudioLocalArtistMediaModelFolder(AudioLocalModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/Artist/Songs"
    table = "artists_media_ordered"
    has_onthego = True
    only_children_otg = True

#     create_stmt = "CREATE TEMP TABLE %s AS " \
#                   "SELECT audios.*,files.path FROM audios,files,audio_albums " \
#                   "WHERE audios.id = files.id AND files.dtime = 0 AND " \
#                   "audios.album_id = audio_albums.id AND " \
#                   "audio_albums.artist_id = ? ORDER BY audios.title COLLATE NOCASE"
    create_stmt = "CREATE TEMP TABLE %s AS " \
                  "SELECT audios.*,files.path FROM audios,files " \
                  "WHERE audios.id = files.id AND files.dtime = 0 AND " \
                  "audios.artist_id = ? ORDER BY audios.title COLLATE NOCASE"

    def _setup_tmp_table(self):
        c = self.create_stmt % self.table
        cur = db.get_cursor()
        cur.execute(self.clear_stmt % self.table)
        cur.execute(c, (self.parent.id,))
        self.commit_changes()


class AudioLocalArtistMenuModelFolder(ModelFolder):
    has_onthego = True
    terra_type = "Model/Folder/Media/Audio/Local/Artist/SubMenu"

    def __init__(self, artist, cur=None, parent=None):
        self.id = artist[0]
        self.name = artist[1].encode("utf-8")
        ModelFolder.__init__(self, self.name, parent)

    def do_load(self):
        self.artist = Artist()
        self.artist.id = self.id
        AudioLocalArtistMediaModelFolder("All songs", self)
        for album in self.artist.albums:
            AudioLocalAlbumMediaModelFolder(album, self)


class AudioLocalArtistModelFolder(AudioLocalModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/Artist"
    children_list_cls = ListLocalArtists
    table = "artists_ordered"
    create_stmt = "CREATE TEMP TABLE %s AS " \
                  "SELECT DISTINCT audio_artists.id,audio_artists.name " \
                  "FROM audios,files,audio_artists WHERE " \
                  "audios.id = files.id AND files.dtime = 0 AND " \
                  "audios.artist_id = audio_artists.id AND " \
                  "audio_artists.name != '' ORDER BY audio_artists.name " \
                  "COLLATE NOCASE"


class AudioLocalAlbumMediaModelFolder(AudioLocalModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/Album/Songs"
    has_onthego = True
    clause = "audios,files USING (id) WHERE album_id = ? " \
             "AND files.dtime = 0 ORDER BY audios.trackno"

    def __init__(self, album, parent=None):
        aux = Album()
        aux.id = album[0]
        self._aux = aux
        self.id = album[0]
        self.name = to_utf8(album[1])
        self.reset()
        AudioLocalModelFolder.__init__(self, self.name, parent)

    def _create_children_list(self):
        self.children = ListLocalMedia(self, self.clause, (self.id,))

    def reset(self):
        self.cover = self._aux.cover
        self.cover_small = self._aux.cover_small
        self.artist = self._aux.artist

    def _setup_tmp_table(self):
        pass


class AudioLocalAlbumModelFolder(AudioLocalModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/Album"
    children_list_cls = ListLocalAlbums
    table = "albums_ordered"
    create_stmt = "CREATE TEMP TABLE %s AS " \
                  "SELECT DISTINCT audio_albums.id,audio_albums.name FROM " \
                  "audios,files,audio_albums WHERE audios.id = files.id " \
                  "AND files.dtime = 0 AND audios.album_id = " \
                  "audio_albums.id AND audio_albums.name != '' " \
                  "ORDER BY audio_albums.name COLLATE NOCASE"


# We just want to use another controller for the same model folder
class AlbumArtAudioModelFolder(AudioLocalAlbumModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/AlbumArt"


class AudioLocalRatingModelFolder(AudioLocalModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/Rating"
    table = "audios_rating"
    has_onthego = True
    create_stmt = "CREATE TEMP TABLE %s AS " \
                  "SELECT * FROM audios,files USING (id) WHERE " \
                  "files.dtime = 0 AND audios.rating = 5 ORDER BY " \
                  "audios.rating COLLATE NOCASE DESC"


class AudioLocalFavouritesModelFolder(AudioLocalModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/Favourites"
    table = "audios_favourite"
    has_onthego = True
    create_stmt = "CREATE TEMP TABLE %s AS " \
                  "SELECT * FROM audios,files USING (id) WHERE " \
                  "files.dtime = 0 AND audios.playcnt > 0 ORDER " \
                  "BY audios.playcnt COLLATE NOCASE DESC LIMIT 25"


class AudioLocalGenreMediaModelFolder(AudioLocalModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/Genre/Songs"
    table = "genre_media_ordered"
    has_onthego = True
    create_stmt = "CREATE TEMP TABLE %s AS " \
                  "SELECT * FROM audios,files USING(id) WHERE " \
                  "genre_id = ? AND files.dtime = 0 ORDER BY " \
                  "audios.title COLLATE NOCASE"

    def __init__(self, id, genre, parent):
        self.id = id
        self.name = genre.encode("utf-8")
        AudioLocalModelFolder.__init__(self, self.name, parent)

    def _setup_tmp_table(self):
        c = self.create_stmt % self.table
        cur = db.get_cursor()
        cur.execute(self.clear_stmt % self.table)
        cur.execute(c, (self.id,))
        self.commit_changes()


class AudioLocalGenreModelFolder(AudioLocalModelFolder):
    terra_type = "Model/Folder/Media/Audio/Local/Genre"
    table = "genre_ordered"
    create_stmt = "CREATE TEMP TABLE %s AS " \
                  "SELECT DISTINCT audio_genres.id,audio_genres.name " \
                  "FROM audios,files,audio_genres WHERE " \
                  " audios.id = files.id AND files.dtime = 0 AND " \
                  "audio_genres.name != '' ORDER BY audio_genres.name " \
                  "COLLATE NOCASE"

    def _create_children_list(self):
        pass # use default list for children

    def do_load(self):
        try:
            self._setup_tmp_table()
            rows = db.execute("SELECT * FROM genre_ordered")
            for row in rows:
                AudioLocalGenreMediaModelFolder(row[0], row[1], self)
        except Exception:
            pass

    def do_unload(self):
        # XXX: since we use the default list, we clear the
        # children the normal way
        ModelFolder.do_unload(self)


# XXX: Activate onthego for Browse By Folder in Audio. Probably a better way
# to do this is having a audio/fs.py file. Until then, we set what we need
# when loading Audio Models.
AudioDBDirectoryModel.has_onthego = True

class AudioLocalBrowse(ModelFolder):
    # This class is not exported but the terra_type is important for
    # getting the right Controller.
    terra_type = "Model/Folder/Directory/Media/Audio"
    child_cls = mger.get_class("Model/Folder/Directory/Media/Audio")
    sys_props = mger.get_class("SystemProperties")()
    has_onthego = True

    def do_load(self):
        prefs = PluginPrefs("scanned_folders")
        if not prefs:
            return

        for path in prefs.get("audio", []):
            if os.path.exists(path):
                name = self.sys_props.path_alias_get(path)
                if not name:
                    name = path
                self.child_cls(path, name, self)


class MyMusicSubMenuModelFolder(Task, ModelFolder):
    terra_type = "Model/Folder/Task/Audio/Local"
    terra_task_type = "Task/Audio/Local"

    def __init__(self, parent):
        Task.__init__(self)
        ModelFolder.__init__(self, "My music", parent)

    def do_load(self):
        AudioLocalAllModelFolder("All songs", self)
        AudioLocalRandomModelFolder("Play random", self)
        AudioLocalArtistModelFolder("Artists", self)
        AudioLocalAlbumModelFolder("Albums", self)
        AlbumArtAudioModelFolder("Cover arts", self)
        AudioLocalPlaylistModelFolder("Playlists", self)
        AudioLocalRatingModelFolder("Top rated", self)
        AudioLocalFavouritesModelFolder("Most played songs", self)
        AudioLocalGenreModelFolder("Genres", self)
        AudioLocalBrowse("Browse by folder", self)

