/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mirco Müller <mirco.mueller@canonical.com>
 *
 */

#include <glib.h>
#include <stdlib.h>
#include <unistd.h>
#include <clutk/clutk.h>
#include <cairo/cairo.h>

static void test_surface_blur (void);

void
test_surface_blur_create_suite (void)
{
#define DOMAIN "/SurfaceBlur"

  g_test_add_func (DOMAIN"/surface-blur", test_surface_blur);
}

void
draw_round_rect (cairo_t* cr,
		 gdouble  aspect,        // aspect-ratio
		 gdouble  x,             // top-left corner
		 gdouble  y,             // top-left corner
		 gdouble  corner_radius, // "size" of the corners
		 gdouble  width,         // width of the rectangle
		 gdouble  height)        // height of the rectangle
{
  gdouble radius = corner_radius / aspect;

  // top-left, right of the corner
  cairo_move_to (cr, x + radius, y);

  // top-right, left of the corner
  cairo_line_to (cr,
                 x + width - radius,
                 y);

  // top-right, below the corner
  cairo_arc (cr,
             x + width - radius,
             y + radius,
             radius,
             -90.0f * G_PI / 180.0f,
             0.0f * G_PI / 180.0f);

  // bottom-right, above the corner
  cairo_line_to (cr,
                 x + width,
                 y + height - radius);

  // bottom-right, left of the corner
  cairo_arc (cr,
             x + width - radius,
             y + height - radius,
             radius,
             0.0f * G_PI / 180.0f,
             90.0f * G_PI / 180.0f);

  // bottom-left, right of the corner
  cairo_line_to (cr,
                 x + radius,
                 y + height);

  // bottom-left, above the corner
  cairo_arc (cr,
             x + radius,
             y + height - radius,
             radius,
             90.0f * G_PI / 180.0f,
             180.0f * G_PI / 180.0f);

  // top-left, right of the corner
  cairo_arc (cr,
             x + radius,
             y + radius,
             radius,
             180.0f * G_PI / 180.0f,
             270.0f * G_PI / 180.0f);
}

static void
test_surface_blur (void)
{
  cairo_t* cr = NULL;
  cairo_surface_t* surface = NULL;

  surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32, 100, 100);
  g_assert (cairo_surface_status (surface) == CAIRO_STATUS_SUCCESS);
  cr = cairo_create (surface);
  g_assert (cairo_status (cr) == CAIRO_STATUS_SUCCESS);

  cairo_scale (cr, 1.0f, 1.0f);
  cairo_set_source_rgba (cr, 1.0f, 1.0f, 1.0f, 1.0f);
  cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
  cairo_paint (cr);

  cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
  cairo_set_source_rgba (cr, 0.0f, 0.0f, 0.0f, 1.0f);
  draw_round_rect (cr, 1.0f, 10.0f, 10.0f, 5.0f, 80.0f, 80.0f);
  cairo_set_line_width (cr, 5.0f);
  cairo_stroke (cr);

  cairo_surface_write_to_png (surface, "./test-surface-blur-before.png");
  ctk_surface_blur (surface, 10);
  cairo_surface_write_to_png (surface, "./test-surface-blur-after.png");

  cairo_destroy (cr);
  cairo_surface_destroy (surface);
}

