/*
 * Accelerometer Driver Test
 * Copyright (C) 2008  Intel Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/**
 * @file accel_test.c
 *
 * Accelerometer driver test
 */

#include <pthread.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include "accelio.h"

pthread_t thread_id;
int device_id;

void *accel_event_handler(void *arg)
{
	struct accel_raw_data data;
	int size = sizeof(data);
	int result = 0;

	printf("start to read acceleration\n");
	while (size == (result = read(device_id, &data, size))) {
		printf("Acceleration is {%d, %d, %d}\n", data.accel_raw_x,
			data.accel_raw_y, data.accel_raw_z);
	}

	if (result == 0) {
		printf("Accel device was closed\n");
	} else {
		printf("Acceleration read error: %s\n", strerror(errno));
	}
	return NULL;
}

int main()
{
	int sense;
	int select;

	device_id =	open("/proc/acpi/accel", O_RDWR | O_SYNC);
	if (-1 == device_id) {
		printf("Can't open the device: %s\n", strerror(errno));
		return 1;
	}

	/* test1: start accelerometer */
	if (-1 == ioctl(device_id, IOCTL_ACCEL_START)) {
		printf("Start Accel failed: %s\n", strerror(errno));
	} else {
		printf("test1: start accelerometer PASS\n");
	}

	/* test2: stop accelerometer */
	if (-1 == ioctl(device_id, IOCTL_ACCEL_STOP)) {
		printf("Stop Accel failed: %s\n", strerror(errno));
	} else {
		printf("test2: stop accelerometer PASS\n");
	}

	/* test3: set sense to 10 when stopped */
	sense = 10;
	if (-1 == ioctl(device_id, IOCTL_ACCEL_SET_SENSE, sense)) {
		printf("Set sense failed when stopped: %s\n",  strerror(errno));
	} else {
		printf("test3: set sense to 10 when stopped PASS\n");
	}

	/* test4: set g-select to a wrong value when stopped*/
	select = 2;
	if (0 == ioctl(device_id, IOCTL_ACCEL_SET_G_SELECT, select)) {
		printf("Set g-select to a wrong value success, but it's BUG!\n");
	} else {
		printf("test4: set g-select to a wrong value when stopped PASS\n");
	}

	/* test5: start accelerometer again after stop it once */
	if (-1 == ioctl(device_id, IOCTL_ACCEL_START)) {
		printf("Start Accel failed: %s\n", strerror(errno));
	} else {
		printf("test5: start accelerometer again after stop it once PASS\n");
	}

	/* test6: set sense to a wrong value */
	sense = 200;
	if (0 == ioctl(device_id, IOCTL_ACCEL_SET_SENSE, sense)) {
		printf("Set sense to a wrong value success, but it is an bug!\n");
	} else {
		printf("test6: set sense to a wrong value PASS\n");
	}

	/* test7: set sense to 5 when started */
	sense = 5;
	if (-1 == ioctl(device_id, IOCTL_ACCEL_SET_SENSE, sense)) {
		printf("Set sense failed when started: %s\n",  strerror(errno));
	} else {
		printf("test7: set sense to 5 when started PASS\n");
	}

	/* test8: set g-select to 6g when started */
	select = 1;
	if (-1 == ioctl(device_id, IOCTL_ACCEL_SET_G_SELECT, select)) {
		printf("Set g-select failed when started: %s\n", strerror(errno));
	} else {
		if (-1 == ioctl(device_id, IOCTL_ACCEL_GET_G_SELECT, &select)) {
			printf("Get g-select failed when started: %s\n", strerror(errno));
		} else {
			if (1 != select) {
				printf("g-select is not correct\n");
			} else {
				printf("test8: set g-select to 6g when started PASS\n");
			}
		}
	}

	/* test9: get acceleration data for 6g */
	printf("test9: get acceleration data for 6g\n");
	printf("It is going to run for one minutes\n");
	printf("Please tilt unit to generate acceleration\n");
	printf("Please check if the acceleration data are correct\n");
	if (pthread_create(&thread_id, NULL, accel_event_handler, NULL)) {
		printf("Can't create thread: %s\n", strerror(errno));
		return 1;
	}

	sleep(60);

	/* test10: set sense to 10 for g-select 1.5g */
	sense = 10;
	if (-1 == ioctl(device_id, IOCTL_ACCEL_SET_SENSE, sense)) {
		printf("Set sense failed when started: %s\n",  strerror(errno));
	} else {
		printf("test10: set sense to 10 when started PASS\n");
	}

	/* test11: set g-select to 1.5g when started */
	select = 0;
	if (-1 == ioctl(device_id, IOCTL_ACCEL_SET_G_SELECT, select)) {
		printf("Set g-select failed when started: %s\n", strerror(errno));
	} else {
		if (-1 == ioctl(device_id, IOCTL_ACCEL_GET_G_SELECT, &select)) {
			printf("Get g-select failed when started: %s\n", strerror(errno));
		} else {
			if (0 != select) {
				printf("g-select is not correct\n");
			} else {
				printf("test11: set g-select to 1.5g when started PASS\n");
			}
		}
	}

	/* test12: get acceleration data for 1.5g */
	printf("test12: get acceleration data\n");
	printf("It is going to run for one minutes\n");
	printf("Please tilt unit to generate acceleration\n");
	printf("Please check if the acceleration data are correct\n");

	sleep(60);
	printf("closing device\n");
	close(device_id);
	printf("waiting thread exit\n");
	pthread_join(thread_id, NULL);

	return 0;
}
