/*
 * $Id: sig_shugart_bus.c,v 1.9 2009-01-27 17:06:42 potyra Exp $
 *
 * Copyright (C) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"
#include <assert.h>
#include <stdio.h>
#include <string.h>

#include "glue-shm.h"

#include "sig_shugart_bus.h"

/* Controller -> Drive */

void
sig_shugart_bus_motor_set(
	struct sig_shugart_bus *b,
	void *s,
	unsigned int drive,
	unsigned int val
)
{
	assert(b);

	if (b->drive[drive].f
	 && b->drive[drive].f->motor_set) {
		(*b->drive[drive].f->motor_set)(b->drive[drive].s, val);
	}
}

void
sig_shugart_bus_select_set(
	struct sig_shugart_bus *b,
	void *s,
	unsigned int drive,
	unsigned int val
)
{
	assert(b);
	if (b->drive[drive].f
	 && b->drive[drive].f->select_set) {
		(*b->drive[drive].f->select_set)(b->drive[drive].s, val);
	}
}

void
sig_shugart_bus_hds_set(struct sig_shugart_bus *b, void *s, unsigned int val)
{
	unsigned int drive;

	assert(b);

	for (drive = 0; drive < 4; drive++) {
		if (! b->drive[drive].f
		 || ! b->drive[drive].f->hds_set) {
			continue;
		}

		(*b->drive[drive].f->hds_set)(b->drive[drive].s, val);
	}
}

void
sig_shugart_bus_step_in(struct sig_shugart_bus *b, void *s)
{
	unsigned int drive;

	assert(b);

	for (drive = 0; drive < 4; drive++) {
		if (! b->drive[drive].f
		 || ! b->drive[drive].f->step_in) {
			continue;
		}

		(*b->drive[drive].f->step_in)(b->drive[drive].s);
	}
}

void
sig_shugart_bus_step_out(struct sig_shugart_bus *b, void *s)
{
	unsigned int drive;

	assert(b);

	for (drive = 0; drive < 4; drive++) {
		if (! b->drive[drive].f
		 || ! b->drive[drive].f->step_out) {
			continue;
		}

		(*b->drive[drive].f->step_out)(b->drive[drive].s);
	}
}

void
sig_shugart_bus_writeid(struct sig_shugart_bus *b, void *s,
		unsigned char cyl, unsigned char sec)
{
	unsigned int drive;

	assert(b);

	for (drive = 0; drive < 4; drive++) {
		if (! b->drive[drive].f
		 || ! b->drive[drive].f->writeid) {
			continue;
		}

		(*b->drive[drive].f->writeid)(b->drive[drive].s,
				cyl, sec);
	}
}

void
sig_shugart_bus_writedata(
	struct sig_shugart_bus *b,
	void *s,
	unsigned char *buf,
	unsigned int bufsize
)
{
	unsigned int drive;

	assert(b);

	for (drive = 0; drive < 4; drive++) {
		if (! b->drive[drive].f
		 || ! b->drive[drive].f->writedata) {
			continue;
		}

		(*b->drive[drive].f->writedata)(b->drive[drive].s,
				buf, bufsize);
	}
}

void
sig_shugart_bus_read_start(struct sig_shugart_bus *b, void *s)
{
	unsigned int drive;

	assert(b);

	for (drive = 0; drive < 4; drive++) {
		if (! b->drive[drive].f
		 || ! b->drive[drive].f->read_start) {
			continue;
		}

		(*b->drive[drive].f->read_start)(b->drive[drive].s);
	}
}

/* Drive -> Controller */

void
sig_shugart_bus_index_set(struct sig_shugart_bus *b, void *s,
		unsigned int val)
{
	assert(b);
	assert(b->ncontrollers == 1);
	assert(b->controller.f);
	assert(b->controller.f->index_set);

	(*b->controller.f->index_set)(b->controller.s, val);
}

void
sig_shugart_bus_wp_set(struct sig_shugart_bus *b, void *s,
		unsigned int val)
{
	assert(b);
	assert(b->ncontrollers == 1);
	assert(b->controller.f);
	assert(b->controller.f->wp_set);

	(*b->controller.f->wp_set)(b->controller.s, val);
}

void
sig_shugart_bus_diskchanged_set(struct sig_shugart_bus *b, void *s,
		unsigned int val)
{
	assert(b);
	assert(b->ncontrollers == 1);
	assert(b->controller.f);
	assert(b->controller.f->diskchanged_set);

	(*b->controller.f->diskchanged_set)(b->controller.s, val);
}

void
sig_shugart_bus_readid(
	struct sig_shugart_bus *b,
	void *s,
	unsigned char cyl,
	unsigned char sec
)
{
	assert(b);
	assert(b->ncontrollers == 1);
	assert(b->controller.f);
	assert(b->controller.f->readid);

	(*b->controller.f->readid)(b->controller.s, cyl, sec);
}

void
sig_shugart_bus_readdata(
	struct sig_shugart_bus *b,
	void *s,
	unsigned char *buf,
	unsigned int bufsize
)
{
	assert(b);
	assert(b->ncontrollers == 1);
	assert(b->controller.f);
	assert(b->controller.f->readdata);

	(*b->controller.f->readdata)(b->controller.s, buf, bufsize);
}

void
sig_shugart_bus_connect_controller(
	struct sig_shugart_bus *b,
	void *s,
	const struct sig_shugart_bus_funcs *f
)
{
	assert(b);
	assert(b->ncontrollers == 0);
	assert(b->type == SIG_GEN_SHUGART_BUS);

	b->controller.s = s;
	b->controller.f = f;
	b->ncontrollers = 1;
}

void
sig_shugart_bus_connect_drive(
	struct sig_shugart_bus *b,
	void *s,
	unsigned int drive,
	const struct sig_shugart_bus_funcs *f
)
{
	assert(/* 0 <= drive && */ drive < 4);
	assert(b);
	assert(! b->drive[drive].s);
	assert(! b->drive[drive].f);
	assert(b->type == SIG_GEN_SHUGART_BUS);

	b->drive[drive].s = s;
	b->drive[drive].f = f;
}

struct sig_shugart_bus *
sig_shugart_bus_init(const char *name, int nr)
{
	struct sig_shugart_bus *b;
	char n[1000];
	int i;

	b = shm_map(name, nr, sizeof(*b), 0);
	b->type = SIG_GEN_SHUGART_BUS;

	sprintf(n, "%s-trk0", name);
	b->trk0 = sig_boolean_or_init(n, nr);
	sprintf(n, "%s-dskchg", name);
	b->dskchg = sig_boolean_or_init(n, nr);

	for(i=0; i < 4; i++) {
		b->drive[i].s = (void *) 0;
		b->drive[i].f = (struct sig_shugart_bus_funcs *) 0;
	}
	b->ncontrollers = 0;

	return b;
}

void
sig_shugart_bus_create(const char *name, int nr)
{
	char n[1000];

	shm_create(name, nr, sizeof(struct sig_shugart_bus));

	sprintf(n, "%s-trk0", name);
	sig_boolean_or_create(n, nr);
	sprintf(n, "%s-dskchg", name);
	sig_boolean_or_create(n, nr);
}

void
sig_shugart_bus_destroy(const char *name, int nr)
{
	char n[1000];

	shm_destroy(name, nr);

	sprintf(n, "%s-trk0", name);
	sig_boolean_or_destroy(n, nr);
	sprintf(n, "%s-dskchg", name);
	sig_boolean_or_destroy(n, nr);
}
