/*
 * Copyright (C) 2009-2010 Freescale Semiconductor, Inc. All rights reserved.
 *
 */
 
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    video_surface.c
 *
 * Description:    Implementation for ipu lib based render service.
 *
 * Portability:    This code is written for Linux OS.
 */  
 
/*
 * Changelog: 
 *
 */

/*=============================================================================
                            INCLUDE FILES
=============================================================================*/
#include <stdio.h>
#include <unistd.h>
#include <stdint.h>

#include <semaphore.h>
#include <fcntl.h>

#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/time.h>

#include "mxc_ipu_hl_lib.h"
#include "video_surface.h"

#include "fsl_debug.h"

#include "vss_common.h"

#undef VSS_DAEMON
/*=============================================================================
                             STATIC VARIABLES
=============================================================================*/
VSLock * 	gVSlock = NULL;
VideoSurfacesControl * gVSctl = NULL;
static VideoSurface * gvslocal = NULL;


VSFlowReturn _configMasterVideoSurface(void * vshandle, void  * config);

VSFlowReturn _configMasterVideoLayer(void * vshandle, void  * config);


static ConfigHandleEntry gConfigHandleTable[] = {
    {CONFIG_MASTER_PARAMETER, sizeof(DestinationFmt), _configMasterVideoSurface},
    {CONFIG_LAYER, 0, _configMasterVideoLayer},
    {CONFIG_INVALID, 0, NULL}
};


/*=============================================================================
                            LOCAL FUNCTIONS
=============================================================================*/
#ifdef METHOD2
static void _initVDIPUTask(VideoDevice * vd)
{
    ipu_lib_input_param_t * input = &vd->copytask.input;
	ipu_lib_output_param_t * output = &vd->copytask.output;

    memset(input, 0, sizeof(ipu_lib_input_param_t));
    memset(output, 0, sizeof(ipu_lib_output_param_t));
    
    input->width = input->input_crop_win.win_w = output->width = output->output_win.win_w =  vd->resx;
    input->height= input->input_crop_win.win_h = output->height= output->output_win.win_h =  vd->resy;

    input->fmt = output->fmt = vd->fmt;
}

static void _kickVDCopy(VideoDevice * vd)
{
    IPUTaskOne * itask = &vd->copytask;

    if (vd->renderidx==0){
        itask->input.user_def_paddr[0] = vd->bufaddr[1];
        itask->output.user_def_paddr[0]=vd->bufaddr[0];
    }else{
        itask->input.user_def_paddr[0] = vd->bufaddr[0];
        itask->output.user_def_paddr[0]=vd->bufaddr[1];
    }

    KICK_IPUTASKONE(itask);
}
#endif    

static int
_checkOnDevice(VideoDevice * vd)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);

    int reconfig = ((vd->cnt==1)?1:0);
    Rect rect, *prect;
    VideoSurface * vs = DEVICE2HEADSURFACE(vd);
    rect = vs->adjustdesrect;

    while(vs=NEXTSURFACE(vs)){
        prect = &vs->adjustdesrect;
        if (rect.left>prect->left){
            rect.left = prect->left;
        }
        if (rect.right<prect->right){
            rect.right = prect->right;
        }        
        if (rect.top>prect->top){
            rect.top = prect->top;
        }       
        if (rect.bottom<prect->bottom){
            rect.bottom = prect->bottom;
        }
    }
    
    prect = &vd->disp;
    if (  (rect.left!=prect->left)
        ||(rect.right!=prect->right)
        ||(rect.top!=prect->top)
        ||(rect.bottom!=prect->bottom)){
        reconfig=1;
    }

    if (reconfig){
        *prect = rect;
    }
    return reconfig;
}

static void
_reconfigAllVideoSurfaces(VideoDevice * vd)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);

    VideoSurface * vs = DEVICE2HEADSURFACE(vd);
    while(vs){
        _initVSIPUTask(vs);
        vs=NEXTSURFACE(vs);
    }
}





void
_initVSIPUTask(VideoSurface * surf)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);

    VideoDevice * vd = SURFACE2DEVICE(surf);
    ipu_lib_input_param_t * input = &surf->itask.input;

    Rect * rect = &(surf->srcfmt.croprect.win), *desrect;
    
    desrect = &(surf->adjustdesrect);
   
    input->fmt = surf->srcfmt.fmt;
    input->width = surf->srcfmt.croprect.width;
    input->height = surf->srcfmt.croprect.height;

    if (surf->outside==0){
        input->input_crop_win.pos.x = rect->left;
        input->input_crop_win.pos.y = rect->top;
        input->input_crop_win.win_w = RECT_WIDTH(rect);
        input->input_crop_win.win_h = RECT_HEIGHT(rect);
    }else{
        /* output outside of screen, need crop in input */
        int xl=0, xr=0, xt=0, xb=0;
        Rect * origrect = &surf->desfmt.rect;
        if (surf->outside&VS_LEFT_OUT){
            xl = (DEVICE_LEFT_EDGE-origrect->left)*RECT_WIDTH(rect)/RECT_WIDTH(origrect);
            ALIGNLEFT8(xl);
        }
        if (surf->outside&VS_RIGHT_OUT){
            xr = (origrect->right-vd->resX)*RECT_WIDTH(rect)/RECT_WIDTH(origrect);
            ALIGNLEFT8(xr);
        }
        if (surf->outside&VS_TOP_OUT){
            xt = (DEVICE_TOP_EDGE-origrect->top)*RECT_HEIGHT(rect)/RECT_HEIGHT(origrect);
            ALIGNLEFT8(xt);
        }
        if (surf->outside&VS_BOTTOM_OUT){
            xb = (origrect->bottom-vd->resY)*RECT_HEIGHT(rect)/RECT_HEIGHT(origrect);
            ALIGNLEFT8(xb);
        }
        
        input->input_crop_win.pos.x = rect->left+xl;
        input->input_crop_win.pos.y = rect->top+xt;
        input->input_crop_win.win_w = RECT_WIDTH(rect)-xl-xr;
        input->input_crop_win.win_h = RECT_HEIGHT(rect)-xt-xb;
        
    }

    input->user_def_paddr[0] = input->user_def_paddr[1] = 0;
    
    ipu_lib_output_param_t * output = &surf->itask.output;
    
    output->fmt = vd->fmt;
    output->width = vd->disp.right-vd->disp.left;
    output->height = vd->disp.bottom-vd->disp.top;
    output->output_win.pos.x = desrect->left-vd->disp.left;
    output->output_win.pos.y = desrect->top-vd->disp.top;
    output->output_win.win_w = desrect->right-desrect->left;
    output->output_win.win_h = desrect->bottom-desrect->top;
    output->user_def_paddr[0] = output->user_def_paddr[1] = 0;

}

static void 
_addVideoSurface2Device(VideoDevice * vd, VideoSurface * vs)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);
    VideoSurface * pvs = DEVICE2HEADSURFACE(vd);

    vs->nextid=0;

    if (pvs){
        while(NEXTSURFACE(pvs))
            pvs=NEXTSURFACE(pvs);
        SET_NEXTSURFACE(pvs, vs);
    }else{
        SET_DEVICEHEADSURFACE(vd,vs);
    }
}

static void 
_removeVideoSurfaceFromDevice(VideoDevice * vd, VideoSurface * vs)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);
    VideoSurface * pvs = DEVICE2HEADSURFACE(vd);
    if (pvs==vs){
        SET_DEVICEHEADSURFACE(vd,NEXTSURFACE(vs));
    }else{
        while(NEXTSURFACE(pvs)!=vs){
            pvs=NEXTSURFACE(pvs);
        }
        SET_NEXTSURFACE(pvs,NEXTSURFACE(vs));
    }
}


static void
_refreshOnDevice(VideoDevice * vd)
{
    VideoSurface * vs = DEVICE2HEADSURFACE(vd);
    Updated update;
    while(vs){
        vs->rendmask = 0;
        _renderSuface(vs, vd,&update);
        vs=NEXTSURFACE(vs);
    }
    _FlipOnDevice(vd);
}

int _adjustDestRect(Rect * rect, VideoDevice * vd)
{
    int outside = 0;
    if (rect->left<DEVICE_LEFT_EDGE){
        outside |= VS_LEFT_OUT;
        rect->left=DEVICE_LEFT_EDGE;
    }
    if (rect->top<DEVICE_TOP_EDGE){
        outside |= VS_TOP_OUT;
        rect->top=DEVICE_TOP_EDGE;
    }
    if (rect->right>vd->resX){
        outside |= VS_RIGHT_OUT;
        rect->right=vd->resX;
    }
    if (rect->bottom>vd->resY){
        outside |= VS_BOTTOM_OUT;
        rect->bottom=vd->resY;
    }

    ALIGNRIGHT8(rect->left);
    ALIGNRIGHT8(rect->top);
    ALIGNLEFT8(rect->right);
    ALIGNLEFT8(rect->bottom);

    
    VS_MESSAGE("adjust win"WIN_FMT"\n", WIN_ARGS(rect));
    

    return outside;
}








VSFlowReturn 
_configMasterVideoLayer(void * vshandle, void  * config)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);

    VideoSurface * vs;
    vs = (VideoSurface *)vshandle;

    if (NEXTSURFACE(vs)==NULL)
        return VS_FLOW_OK;

    VideoDevice * vd=SURFACE2DEVICE(vs);

      
    VS_LOCK(gVSlock);

    _removeVideoSurfaceFromDevice(vd,vs);
    _addVideoSurface2Device(vd,vs);
    
    _refreshOnDevice(vd);

    VS_UNLOCK(gVSlock);
    VS_FLOW("Fun %s out\n", __FUNCTION__);

    return VS_FLOW_OK;

}




VSFlowReturn 
_configMasterVideoSurface(void * vshandle, void  * config)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);
    DestinationFmt * des = (DestinationFmt *)config;

    VideoSurface * vs;
    SourceFmt src;
    vs = (VideoSurface *)vshandle;

    VS_MESSAGE("reconfig win from "WIN_FMT" to "WIN_FMT"\n", WIN_ARGS(&vs->desfmt.rect), WIN_ARGS(&des->rect));

    VideoDevice * vd=SURFACE2DEVICE(vs);

      
    VS_LOCK(gVSlock);
    vs->desfmt = *des;
    vs->outside = _adjustDestRect(&des->rect, vd);
    vs->adjustdesrect = des->rect;

    if (NEXTSURFACE(vs)){
        _removeVideoSurfaceFromDevice(vd,vs);
        _addVideoSurface2Device(vd,vs);
    }

    
//#ifndef VSS_DAEMON
    if (_checkOnDevice(vd)){
        _reconfigAllVideoSurfaces(vd);
        _setDeviceConfig(vd);
    }else{
        _initVSIPUTask(vs);
    }
    if (vd->setalpha)
        _setAlpha(vd);
    
    _refreshOnDevice(vd);
//#endif

    VS_UNLOCK(gVSlock);
    VS_FLOW("Fun %s out\n", __FUNCTION__);

    return VS_FLOW_OK;

}



void 
_destroyVideoSurface(void * vshandle, int force)
{

    VideoSurface * vs, *vs1;
    
    VideoDevice * vd;
    vs = (VideoSurface *)vshandle;

    if (vs==NULL)
        return;
    
    VS_LOCK(gVSlock);
    
    if (force==0){
        vs1 = gvslocal;
        if (vs1==vs){
            gvslocal = vs->next;
        }else{
            while(vs1->next!=vs)
                vs1=vs1->next;
                vs1->next = vs->next;
        }
    }
        
    vd = SURFACE2DEVICE(vs);

    _removeVideoSurfaceFromDevice(vd,vs);

    vd->cnt--;
//#ifndef VSS_DAEMON
    if (DEVICE2HEADSURFACE(vd)==NULL){
        _closeDevice(vd);
    }else{
        if (_checkOnDevice(vd)){
            _reconfigAllVideoSurfaces(vd);
            _setDeviceConfig(vd);
        }
    }
    
    if (vd->setalpha)
        _setAlpha(vd);
//#endif


    VS_MESSAGE("Videosurface %d destroied, force=%d!\n", vs->id, force);
    memset(vs, 0, sizeof(VideoSurface));
    
    vs->status = VS_STATUS_IDLE;
    
    VS_UNLOCK(gVSlock);
}



/*=============================================================================
FUNCTION:           createVideoSurface

DESCRIPTION:        This function create a video surface.
==============================================================================*/
void * 
createVideoSurface(int fbnum, SourceFmt * src, DestinationFmt * des)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);

    VideoSurfacesControl * vc;
    VideoSurface * vs = NULL;
    VideoSurface *vs1;
    VideoDevice * vd;
    int i;
    
    if ((des==NULL)||(src==NULL)){
        VS_ERROR("%s: parameters error!\n", __FUNCTION__);
        goto err;
    }

    if (gVSctl==NULL){
#ifndef VSS_DAEMON
        gVSlock = _getAndLockVSLock(VS_IPC_CREATE);
#else
        gVSlock = _getAndLockVSLock(0);
#endif
        if (gVSlock==NULL){
            VS_ERROR("Can not create/open ipc semphone!\n");
            goto err;
        }

#ifndef VSS_DAEMON
        gVSctl = _getVSControl(VS_IPC_CREATE);
#else
        gVSctl = _getVSControl(0);
#endif
        if (gVSctl==NULL){
            VS_ERROR("Can not create/open ipc sharememory!\n");
            VS_UNLOCK(gVSlock);
            goto err;
        }
    }else{

        VS_LOCK(gVSlock);
    }
    vc = gVSctl;

    for (i=0;i<MAX_VIDEO_SURFACE;i++){
        if (vc->surfaces[i].status==VS_STATUS_IDLE){
            break;
        }
    }
    
    if (i==MAX_VIDEO_SURFACE){
        VS_UNLOCK(gVSlock);
        VS_ERROR("%s: max surface support exceeded!\n", __FUNCTION__);
        goto err;
    }

    vs = &vc->surfaces[i];
    vs->id = i+1;
    vs->status = VS_STATUS_VISIBLE;
    vs->devid = 1;
    vs->srcfmt = *src;
    vs->desfmt = *des;
    
    vd = SURFACE2DEVICE(vs);

    vd->fbidx = 2;

#if 0    
    if (gFBDesc[ID2INDEX(vd->id)].fb_fd==0){
        int fd = open(gFBDesc[ID2INDEX(vd->id)].devname, O_RDWR, 0);
        
        if (fd<=0){
            VS_UNLOCK(gVSlock);
            VS_ERROR("%s: max surface support exceeded!\n", __FUNCTION__);
            goto err;
        }
        gFBDesc[ID2INDEX(vd->id)].fb_fd = fd;
    }
#endif

#ifndef VSS_DAEMON
    if (vd->init==0){
        _initVideoDevice(vd);
        vd->init=1;
    }
#endif

    vs->outside = _adjustDestRect(&des->rect, vd);
    vs->adjustdesrect = des->rect;


    VS_MESSAGE("Create Videosurface:fmt "FOURCC_FMT", windows"WIN_FMT", padded %dx%d\n", 
        FOURCC_ARGS(src->fmt), WIN_ARGS(&src->croprect.win), src->croprect.width, src->croprect.height);

    vs->next = gvslocal;
    gvslocal = vs;

    vd->cnt++;
    
    if (vd->cnt==1){
        vd->id = INDEX2ID(DEFAULT_FB_DEV_INDEX);
        SET_DEVICEHEADSURFACE(vd, vs);
        
    }else{
        _addVideoSurface2Device(vd, vs);
    }
//#ifndef VSS_DAEMON
    if (_checkOnDevice(vd)){
        _reconfigAllVideoSurfaces(vd);
        _setDeviceConfig(vd);
    }
    _initVSIPUTask(vs);

    if (vd->cnt==1)
        _openDevice(vd);
    if (vd->setalpha)
        _setAlpha(vd);
//#endif

    VS_UNLOCK(gVSlock);
    VS_FLOW("Fun %s out\n", __FUNCTION__);    
    
    return (void *)vs;
err:
    if (vs){
        memset(vs, 0, sizeof(VideoSurface));
    }
    return NULL;
}



/*=============================================================================
FUNCTION:           destroyVideoSurface

DESCRIPTION:        This function destroy a video surface create before
==============================================================================*/
void 
destroyVideoSurface(void * vshandle)
{
    _destroyVideoSurface(vshandle, 0);
}


/*=============================================================================
FUNCTION:           configVideoSurface

DESCRIPTION:        This function reconfig the specific video surface including 
                    output window size, position and rotation.
==============================================================================*/
VSFlowReturn 
configVideoSurface(void * vshandle, VSConfigID confid, VSConfig * config)
{
    int ret = VS_FLOW_OK;
    ConfigHandleEntry * configentry = gConfigHandleTable;
    if (vshandle==NULL)
        return VS_FLOW_PARAMETER_ERROR;
    
    while(configentry->cid!=CONFIG_INVALID){
        if (configentry->cid==confid){
            if (config->length!=configentry->parameterlen){
                return VS_FLOW_PARAMETER_ERROR;
            }
            if (configentry->handle){
                ret = (*configentry->handle)(vshandle, config->data);
                return ret;
            }
        }
        configentry++;
    }
    return ret;
}


/*=============================================================================
FUNCTION:           render2VideoSurface

DESCRIPTION:        This function render a new frame on specific video surface
                    It also will refresh other video surfaces in same video device
                    automaticallly.
==============================================================================*/
VSFlowReturn 
render2VideoSurface(void * vshandle, SourceFrame * frame, SourceFmt * srcfmt)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);

    VideoDevice * vd;
    VideoSurface * vsurface, *vsurface1;
    Updated updated;
    struct timeval current;
    Rect * r;

    if ((vshandle==NULL)||(frame==NULL)){
        VS_ERROR("%s: parameters error!\n", __FUNCTION__);
        return VS_FLOW_PARAMETER_ERROR;
    }

    vsurface = (VideoSurface *)vshandle;

    if (vsurface->status == VS_STATUS_INVISIBLE) /* no need to render */
        return VS_FLOW_OK;
    
    vsurface->paddr = frame->paddr;
    vsurface->rendmask = 0;/*clear mask*/
    
    vd = SURFACE2DEVICE(vsurface);
    vd->flag |= 1;


#ifndef VSS_DAEMON
    if (VS_TRY_LOCK(gVSlock)){
        return VS_FLOW_PENDING;
    }
    
    
    gettimeofday(&current, NULL);

    if ((current.tv_sec<vd->timestamp.tv_sec)||
        ((current.tv_sec<vd->timestamp.tv_sec)&&(current.tv_usec<vd->timestamp.tv_usec))){
        goto done;
    }

    current.tv_usec+=MIN_RENDER_INTERVAL_IN_MICROSECOND;
    if (current.tv_usec>1000000){
        current.tv_usec-=1000000;
        current.tv_sec++;
    }
    vd->timestamp = current;

#ifdef METHOD2
    vsurface1 = vsurface;// DEVICE2HEADSURFACE(vd);
#else    
    vsurface1 = DEVICE2HEADSURFACE(vd);
#endif

#ifdef METHOD2
    _kickVDCopy(vd);
#endif

    memset((void *)(&updated), 0, sizeof(Updated));
    while(vsurface1){
        if (_needRender(vsurface1, &updated, vd->renderidx)){
            _renderSuface(vsurface1, vd, &updated);
        }
        vsurface1 = NEXTSURFACE(vsurface1);
    };

    _FlipOnDevice(vd);
    //usleep(10000);
done:
    VS_UNLOCK(gVSlock);


#endif    
    VS_FLOW("Fun %s out\n", __FUNCTION__);
    return VS_FLOW_OK;
err:
    return VS_FLOW_ERROR;
}


/*=============================================================================
FUNCTION:           video_surface_destroy

DESCRIPTION:        This function deconstruct created video surfaces and recycle 
                    all hardware resource.
==============================================================================*/
void __attribute__ ((destructor)) video_surface_destroy(void);
void video_surface_destroy(void)
{
    VS_FLOW("Fun %s in\n", __FUNCTION__);
    VideoSurface * vs = gvslocal, *vsnext;
    while(vs){
        vsnext = vs->next;
        _destroyVideoSurface(vs, 1);
        vs = vsnext;
    }
    
}



