/*
 * indicator-network
 * Copyright 2010-2012 Canonical Ltd.
 *
 * Authors:
 * Antti Kaijanmäki <antti.kaijanmaki@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "android-telephony-service.h"

#include <gio/gio.h>

#define ANDROID_TELEPHONY_SERVICE_DBUS_SERVICE     "com.canonical.Android"
#define ANDROID_TELEPHONY_SERVICE_DBUS_OBJECT_PATH "/com/canonical/android/telephony/Telephony"
#define ANDROID_TELEPHONY_SERVICE_DBUS_INTERFACE   "com.canonical.android.telephony.Telephony"

typedef struct _AndroidTelephonyServicePrivate AndroidTelephonyServicePrivate;

struct _AndroidTelephonyServicePrivate
{
  GDBusProxy *proxy;
  guint       watch_id;

  gboolean                  is_gsm;
  gchar                    *operator_name;
  TelephonyRadioTechnology  radio_technology;
  gboolean                  roaming;
  TelephonySignalLevel      signal_level;
  TelephonyState            state;
};

enum {
  PROP_0,

  
};

enum {
  SIG_0,

  SIG_IS_GSM_CHANGED,
  SIG_OPERATOR_NAME_CHANGED,
  SIG_RADIO_TECHNOLOGY_CHANGED,
  SIG_ROAMING_CHANGED,
  SIG_SIGNAL_LEVEL_CHANGED,
  SIG_STATE_CHANGED,

  SIG_LAST
};
static int signals[SIG_LAST];

#define GET_PRIVATE(o)                                 \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o),                   \
                                ANDROID_TELEPHONY_SERVICE_TYPE,  \
                                AndroidTelephonyServicePrivate))

static void android_telephony_service_dispose (GObject *object);
static void android_telephony_service_finalize(GObject *object);

static void set_is_gsm(AndroidTelephonyService *self, gboolean value);
static void set_operator_name(AndroidTelephonyService *self,
			      const gchar *value);
static void set_radio_technology(AndroidTelephonyService *self,
				 TelephonyRadioTechnology value);
static void set_roaming(AndroidTelephonyService *self, gboolean value);
static void set_signal_level(AndroidTelephonyService *self,
			     TelephonySignalLevel value);
static void set_state(AndroidTelephonyService *self,
		      TelephonyState value);

G_DEFINE_TYPE(AndroidTelephonyService, android_telephony_service, G_TYPE_OBJECT);


static void
proxy_properties_changed_handler(GDBusProxy *proxy,
				 GVariant   *changed_properties,
				 GStrv       invalidated_properties,
				 gpointer    user_data)
{
  AndroidTelephonyService        *self = NULL;
  AndroidTelephonyServicePrivate *priv = NULL;

  GVariantIter *iter = NULL;

  gchar *property;
  GVariant *value;

  self = ANDROID_TELEPHONY_SERVICE(user_data);
  priv = GET_PRIVATE(self);

  g_variant_get(changed_properties, "a{sv}", &iter);

  if (iter == NULL) {
    g_warning("%s: could not get iter", __func__);
    return;
  }

  while(g_variant_iter_loop(iter,
			    "{sv}",
			    &property,
			    &value)) {

    if (g_strcmp0(property, "IsGsm") == 0) {
      set_is_gsm(self, g_variant_get_boolean(value));
    } else if (g_strcmp0(property, "OperatorName") == 0) {
      set_operator_name(self, g_variant_get_string(value, 0));
    } else if (g_strcmp0(property, "RadioTechnology") == 0) {
      set_radio_technology(self, g_variant_get_int32(value));
    } else if (g_strcmp0(property, "Roaming") == 0) {
      set_roaming(self, g_variant_get_boolean(value));
    } else if (g_strcmp0(property, "SignalLevel") == 0) {
      set_signal_level(self, g_variant_get_int32(value));
    } else if (g_strcmp0(property, "State") == 0) {
      set_state(self, g_variant_get_int32(value));
    }
  }

  g_variant_iter_free(iter);
}

static void
proxy_ready_cb(GObject *source_object,
	       GAsyncResult *res,
	       gpointer user_data)
{
  AndroidTelephonyService        *self = NULL;
  AndroidTelephonyServicePrivate *priv = NULL;
  GError *error = NULL;
  GVariant *value = NULL;

  g_debug("%s", __func__);

  self = ANDROID_TELEPHONY_SERVICE(user_data);
  g_assert(self != NULL);
  
  priv = GET_PRIVATE(self);

  priv->proxy = g_dbus_proxy_new_for_bus_finish(res, &error);

  if (error != NULL) {
    priv->proxy = NULL;
    g_critical("%s: Could not create proxy: %s", __func__, error->message);
    g_error_free(error);
    return;
  }

  g_signal_connect(priv->proxy,
		   "g-properties-changed",
		   G_CALLBACK(proxy_properties_changed_handler),
		   self);

  value = g_dbus_proxy_get_cached_property(priv->proxy, "IsGsm");
  if (value == NULL) {
    g_warning("%s: could not get cached IsGsm", __func__);
  } else {
    set_is_gsm(self, g_variant_get_boolean(value));
    g_variant_unref(value);
  }

  value = g_dbus_proxy_get_cached_property(priv->proxy, "OperatorName");
  if (value == NULL) {
    g_warning("%s: could not get cached OperatorName", __func__);
  } else {
    set_operator_name(self, g_variant_get_string(value, 0));
    g_variant_unref(value);
  }

  value = g_dbus_proxy_get_cached_property(priv->proxy, "RadioTechnology");
  if (value == NULL) {
    g_warning("%s: could not get cached RadioTechnology", __func__);
  } else {
    set_radio_technology(self, g_variant_get_int32(value));
    g_variant_unref(value);
  }

  value = g_dbus_proxy_get_cached_property(priv->proxy, "Roaming");
  if (value == NULL) {
    g_warning("%s: could not get cached Roaming", __func__);
  } else {
    set_roaming(self, g_variant_get_boolean(value));
    g_variant_unref(value);
  }

  value = g_dbus_proxy_get_cached_property(priv->proxy, "SignalLevel");
  if (value == NULL) {
    g_warning("%s: could not get cached SignalLevel", __func__);
  } else {
    set_signal_level(self, g_variant_get_int32(value));
    g_variant_unref(value);
  }

  value = g_dbus_proxy_get_cached_property(priv->proxy, "State");
  if (value == NULL) {
    g_warning("%s: could not get cached State", __func__);
  } else {
    set_state(self, g_variant_get_int32(value));
    g_variant_unref(value);
  }
}

static void
service_appeared(GDBusConnection *connection,
		 const gchar *name,
		 const gchar *name_owner,
		 gpointer user_data)
{
  AndroidTelephonyService        *self = NULL;
  AndroidTelephonyServicePrivate *priv = NULL;

  g_debug("service_appeared: called");

  self = ANDROID_TELEPHONY_SERVICE(user_data);
  g_assert(self != NULL);

  priv = GET_PRIVATE(self);

  g_assert(priv->proxy == NULL);

  g_dbus_proxy_new_for_bus(G_BUS_TYPE_SESSION,
			   G_DBUS_PROXY_FLAGS_NONE,
			   NULL,
			   ANDROID_TELEPHONY_SERVICE_DBUS_SERVICE,
			   ANDROID_TELEPHONY_SERVICE_DBUS_OBJECT_PATH,
			   ANDROID_TELEPHONY_SERVICE_DBUS_INTERFACE,
			   NULL,
			   proxy_ready_cb,
			   self);

  g_debug("service_appeared: proxy_new_for_bus called");
}

static void
service_vanished(GDBusConnection *connection,
		 const gchar *name,
		 gpointer user_data)
{
  AndroidTelephonyService        *self;
  AndroidTelephonyServicePrivate *priv;

  g_debug("service_vanished: called");

  self = ANDROID_TELEPHONY_SERVICE(user_data);
  g_assert(self != NULL);

  priv = GET_PRIVATE(self);

  if(priv->proxy == NULL)
    return;

  g_object_unref(priv->proxy);
  priv->proxy = NULL;
}


static void
android_telephony_service_class_init(AndroidTelephonyServiceClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
  GParamSpec *spec;

  g_type_class_add_private(klass, sizeof(AndroidTelephonyServicePrivate));

  gobject_class->dispose      = android_telephony_service_dispose;
  gobject_class->finalize     = android_telephony_service_finalize;
  
  signals[SIG_IS_GSM_CHANGED] =
    g_signal_new("is-gsm-changed",
		 G_TYPE_FROM_CLASS(gobject_class),
		 G_SIGNAL_RUN_FIRST,
		 0,
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__BOOLEAN,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_BOOLEAN);

  signals[SIG_OPERATOR_NAME_CHANGED] =
    g_signal_new("operator-name-changed",
		 G_TYPE_FROM_CLASS(gobject_class),
		 G_SIGNAL_RUN_FIRST,
		 0,
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__VOID,
		 G_TYPE_NONE,
		 0);

  signals[SIG_RADIO_TECHNOLOGY_CHANGED] =
    g_signal_new("radio-technology-changed",
		 G_TYPE_FROM_CLASS(gobject_class),
		 G_SIGNAL_RUN_FIRST,
		 0,
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__INT,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_INT);

  signals[SIG_ROAMING_CHANGED] =
    g_signal_new("roaming-changed",
		 G_TYPE_FROM_CLASS(gobject_class),
		 G_SIGNAL_RUN_FIRST,
		 0,
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__BOOLEAN,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_BOOLEAN);

  signals[SIG_SIGNAL_LEVEL_CHANGED] =
    g_signal_new("signal-level-changed",
		 G_TYPE_FROM_CLASS(gobject_class),
		 G_SIGNAL_RUN_FIRST,
		 0,
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__INT,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_INT);

  signals[SIG_STATE_CHANGED] =
    g_signal_new("state-changed",
		 G_TYPE_FROM_CLASS(gobject_class),
		 G_SIGNAL_RUN_FIRST,
		 0,
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__INT,
		 G_TYPE_NONE,
		 1,
		 G_TYPE_INT);
}

static void
android_telephony_service_init(AndroidTelephonyService *self)
{
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);

  g_debug("android_telephony_service_init: called");

  priv->operator_name = g_strdup("");

  priv->proxy = NULL;
  priv->watch_id = g_bus_watch_name(G_BUS_TYPE_SESSION,
				    ANDROID_TELEPHONY_SERVICE_DBUS_SERVICE,
				    G_BUS_NAME_WATCHER_FLAGS_AUTO_START,
				    service_appeared,
				    service_vanished,
				    self,
				    NULL);
}

static void
android_telephony_service_dispose(GObject *object)
{
  AndroidTelephonyService *self        = ANDROID_TELEPHONY_SERVICE(object);
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);

  g_debug("%s", __func__);

  if (priv->watch_id != 0) {
    g_bus_unwatch_name(priv->watch_id);
    priv->watch_id = 0;
  }

  if (priv->proxy != NULL) {
    g_object_unref(priv->proxy);
    priv->proxy = NULL;
  }

  G_OBJECT_CLASS(android_telephony_service_parent_class)->dispose(object);
}

static void
android_telephony_service_finalize(GObject *object)
{
  AndroidTelephonyService *self = ANDROID_TELEPHONY_SERVICE(object);
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);

  g_free(priv->operator_name);

  G_OBJECT_CLASS(android_telephony_service_parent_class)->finalize(object);
}

AndroidTelephonyService *
android_telephony_service_new()
{
  return g_object_new(ANDROID_TELEPHONY_SERVICE_TYPE, NULL);
}

static void
set_is_gsm(AndroidTelephonyService *self, gboolean value)
{
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);
  
  if (value == priv->is_gsm)
    return;
  
  priv->is_gsm = value;
  g_signal_emit(self,
		signals[SIG_IS_GSM_CHANGED],
		0,
		priv->is_gsm);
}

gboolean
android_telephony_service_get_is_gsm(AndroidTelephonyService *self)
{
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);

  return priv->is_gsm;
}

static void
set_operator_name(AndroidTelephonyService *self, const gchar *value)
{
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);
  
  if (g_strcmp0(value, priv->operator_name) == 0)
    return;
  
  g_free(priv->operator_name);
  priv->operator_name = g_strdup(value);
  
  g_signal_emit(self,
		signals[SIG_OPERATOR_NAME_CHANGED],
		0);
}

const gchar *
android_telephony_service_get_operator_name(AndroidTelephonyService *self)
{
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);

  return priv->operator_name;
}

static void
set_radio_technology(AndroidTelephonyService *self,
		     TelephonyRadioTechnology value)
{
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);
  
  if (value == priv->radio_technology)
    return;

  priv->radio_technology = value;

  g_signal_emit(self,
		signals[SIG_RADIO_TECHNOLOGY_CHANGED],
		0,
		priv->radio_technology);
}

TelephonyRadioTechnology
android_telephony_service_get_radio_technology(AndroidTelephonyService *self)
{
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);

  return priv->radio_technology;
}

static void
set_roaming(AndroidTelephonyService *self, gboolean value)
{
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);
  
  if (value == priv->roaming)
    return;

  priv->roaming = value;
  g_signal_emit(self,
		signals[SIG_ROAMING_CHANGED],
		0,
		priv->roaming);
}

gboolean
android_telephony_service_get_roaming(AndroidTelephonyService *self)
{
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);

  return priv->roaming;
}

static void
set_signal_level(AndroidTelephonyService *self,
		 TelephonySignalLevel value)
{
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);

  if (value == priv->signal_level)
    return;

  priv->signal_level = value;
  
  g_signal_emit(self,
		signals[SIG_SIGNAL_LEVEL_CHANGED],
		0,
		priv->signal_level);
}

TelephonySignalLevel
android_telephony_service_get_signal_level(AndroidTelephonyService *self)
{
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);

  return priv->signal_level;
}

static void
set_state(AndroidTelephonyService *self,
	  TelephonyState value)
{
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);

  if (value == priv->state)
    return;
  
  priv->state = value;

  g_signal_emit(self,
		signals[SIG_STATE_CHANGED],
		0,
		priv->state);
}

TelephonyState
android_telephony_service_get_state(AndroidTelephonyService *self)
{
  AndroidTelephonyServicePrivate *priv = GET_PRIVATE(self);

  return priv->state;
}
