/*
 * arch/arm/mach-dove/mimas.c
 *
 * Wyse Mimas/Titan Board
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2.  This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/irq.h>
#include <linux/mtd/physmap.h>
#include <linux/mtd/nand.h>
#include <linux/timer.h>
#include <linux/ata_platform.h>
#include <linux/mv643xx_eth.h>
#include <linux/i2c.h>
#include <linux/i2c/kb3310.h>
#include <linux/pci.h>
#include <linux/gpio_mouse.h>
#include <linux/spi/spi.h>
#include <linux/spi/orion_spi.h>
#include <linux/spi/flash.h>
#include <linux/i2c/eneec.h>
#include <video/dovefb.h>
#include <video/dovefbreg.h>
#include <mach/dove_bl.h>
#include <plat/i2s-orion.h>
#include <asm/mach-types.h>
#include <asm/setup.h>
#include <asm/gpio.h>
#include <asm/mach/arch.h>
#include <mach/dove.h>
#include <asm/hardware/pxa-dma.h>
#include <plat/orion_nfc.h>
#include <mach/pm.h>
#include <plat/cafe-orion.h>
#include "common.h"
#include "clock.h"
#include "mpp.h"
#include "mach/pm.h"
#include "pmu/mvPmu.h"
#include "pmu/mvPmuRegs.h"
#include "pdma/mvPdma.h"
#include <ctrlEnv/mvCtrlEnvRegs.h>
#include <audio/mvAudioRegs.h>
#include "gpp/mvGppRegs.h"
#include "idt5v49ee503.h"

#define GPIO_MCU_INTRn		(0)
#define GPIO_I2S_CODEC_IRQ	(1)
#define GPIO_MPP_DDR_TERM	(16)
#define GPIO_OFF_CTRL		(8)
#define GPIO_EC_IRQ		(7)
#define GPIO_HUB_RESETn		(9)
#define GPIO_LCM_DCM		(14)
#define GPIO_GP_WLAN_RSTn	(20)
#define GPIO_G_INT		(23)
#define GPIO_HEADSET_DET	(45)

#define GPIO_TITAN_LCM_BL_CTRL	(40)
#define GPIO_TITAN_MIC_DET	(42)

extern int __init pxa_init_dma_wins(struct mbus_dram_target_info *dram);

static unsigned int use_hal_giga = 0;
#ifdef CONFIG_MV643XX_ETH
module_param(use_hal_giga, uint, 0);
MODULE_PARM_DESC(use_hal_giga, "Use the HAL giga driver");
#endif

static unsigned int standby_fix = 1;
module_param(standby_fix, uint, 0);
MODULE_PARM_DESC(standby_fix, "if 1 then CKE and MRESET are connected to MPP4 and MPP6");

static int dvs_enable = 0;
module_param(dvs_enable, int, 1);
MODULE_PARM_DESC(dvs_enable, "if 1 then enable DVS");

extern unsigned int useHalDrivers;
extern char *useNandHal;

enum {
	MIMAS_EVT,
	MIMAS_DVT,
	TITAN_EVT,
	TITAN_DVT,
};

static int mimas_rev = MIMAS_DVT;	/* default to DVT */
static int titan_rev = TITAN_DVT;	/* default to DVT */

/*****************************************************************************
 * LCD
 ****************************************************************************/

/*
 * LCD HW output Red[0] to LDD[0] when set bit [19:16] of reg 0x190
 * to 0x0. Which means HW outputs BGR format default. All platforms
 * uses this controller should enable .panel_rbswap. Unless layout
 * design connects Blue[0] to LDD[0] instead.
 */
static struct dovefb_mach_info lcd0_dmi = {
	.id_gfx			= "GFX Layer 0",
	.id_ovly		= "Video Layer 0",
	.clk_src		= MRVL_EXT_CLK0,
	.clk_name		= "IDT_CLK0",
//	.num_modes		= ARRAY_SIZE(video_modes),
//	.modes			= video_modes,
	.pix_fmt		= PIX_FMT_RGB888PACK,
	.io_pin_allocation	= IOPAD_DUMB24,
	.panel_rgb_type		= DUMB24_RGB888_0,
	.panel_rgb_reverse_lanes= 0,
	.gpio_output_data	= 0,
	.gpio_output_mask	= 0,
	.ddc_i2c_adapter	= 1,
	.ddc_i2c_address	= 0x50,
	.invert_composite_blank	= 0,
	.invert_pix_val_ena	= 0,
	.invert_pixclock	= 0,
	.invert_vsync		= 0,
	.invert_hsync		= 0,
	.panel_rbswap		= 1,
	.active			= 1,
};

static struct dovefb_mach_info lcd0_vid_dmi = {
	.id_ovly		= "Video Layer 0",
//	.num_modes		= ARRAY_SIZE(video_modes),
//	.modes			= video_modes,
	.pix_fmt		= PIX_FMT_RGB888PACK,
	.io_pin_allocation	= IOPAD_DUMB24,
	.panel_rgb_type		= DUMB24_RGB888_0,
	.panel_rgb_reverse_lanes= 0,
	.gpio_output_data	= 0,
	.gpio_output_mask	= 0,
	.ddc_i2c_adapter	= -1,
	.invert_composite_blank	= 0,
	.invert_pix_val_ena	= 0,
	.invert_pixclock	= 0,
	.invert_vsync		= 0,
	.invert_hsync		= 0,
	.panel_rbswap		= 1,
	.active			= 1,
	.enable_lcd0		= 1,
};

static struct dovefb_mach_info lcd1_dmi = {
	.id_gfx			= "GFX Layer 1",
	.id_ovly		= "Video Layer 1",
	.clk_src		= MRVL_PLL_CLK,
	.clk_name		= "LCDCLK",
//	.num_modes		= ARRAY_SIZE(video_modes),
//	.modes			= video_modes,
	.pix_fmt		= PIX_FMT_RGB888PACK,
	.io_pin_allocation	= IOPAD_DUMB24,
	.panel_rgb_type		= DUMB24_RGB888_0,
	.panel_rgb_reverse_lanes= 0,
	.gpio_output_data	= 0,
	.gpio_output_mask	= 0,
	.ddc_i2c_adapter	= 0,
	.ddc_i2c_address	= 0x50,
	.invert_composite_blank	= 0,
	.invert_pix_val_ena	= 0,
	.invert_pixclock	= 0,
	.invert_vsync		= 0,
	.invert_hsync		= 0,
	.panel_rbswap		= 1,
	.active			= 1,
#ifndef CONFIG_FB_DOVE_CLCD
	.enable_lcd0		= 1,
#else
	.enable_lcd0		= 0,
#endif
};

static struct dovefb_mach_info lcd1_vid_dmi = {
	.id_ovly		= "Video Layer 1",
//	.num_modes		= ARRAY_SIZE(video_modes),
//	.modes			= video_modes,
	.pix_fmt		= PIX_FMT_RGB888PACK,
	.io_pin_allocation	= IOPAD_DUMB24,
	.panel_rgb_type		= DUMB24_RGB888_0,
	.panel_rgb_reverse_lanes= 0,
	.gpio_output_data	= 0,
	.gpio_output_mask	= 0,
	.ddc_i2c_adapter	= -1,
	.invert_composite_blank	= 0,
	.invert_pix_val_ena	= 0,
	.invert_pixclock	= 0,
	.invert_vsync		= 0,
	.invert_hsync		= 0,
	.panel_rbswap		= 1,
	.active			= 0,
};

static struct dovedcon_mach_info mimas_dcon_data = {
	.mode	= PORTA_ENABLE | PORTA_LCD0_BYPASS |
		  PORTB_ENABLE | PORTB_LCD1_BYPASS,
};

static struct dovebl_platform_data titan_backlight_data = {
	.default_intensity	= 0xa,
	.gpio_pm_control	= 1,

	.lcd_start	= DOVE_SB_REGS_VIRT_BASE, /* lcd power control reg base. */
	.lcd_end	= DOVE_SB_REGS_VIRT_BASE + GPP_DATA_OUT_REG(0), /* end of reg map. */
	.lcd_offset	= GPP_DATA_OUT_REG(0), /* register offset */
	.lcd_mapped	= 1,		/* va = 0, pa = 1 */
	.lcd_mask	= 0x800,	/* mask, bit[11] */
	.lcd_on		= 0x800,	/* value to enable lcd power */
	.lcd_off	= 0x0,		/* value to disable lcd power */

	.blpwr_start	= DOVE_SB_REGS_VIRT_BASE, /* bl pwr ctrl reg base. */
	.blpwr_end	= DOVE_SB_REGS_VIRT_BASE + GPP_DATA_OUT_REG(0),	/* end of reg map. */
	.blpwr_offset	= GPP_DATA_OUT_REG(0),	/* register offset */
	.blpwr_mapped	= 1,		/* pa = 0, va = 1 */
	.blpwr_mask	= 0x2000,	/* mask, bit[13] */
	.blpwr_on	= 0x2000,	/* value to enable bl power */
	.blpwr_off	= 0x0,		/* value to disable bl power */

	.btn_start	= DOVE_LCD1_PHYS_BASE, /* brightness control reg base. */
	.btn_end	= DOVE_LCD1_PHYS_BASE + 0x1C8, /* end of reg map. */
	.btn_offset	= LCD_CFG_GRA_PITCH, /* register offset */
	.btn_mapped	= 0,		/* pa = 0, va = 1 */
	.btn_mask	= 0xF0000000,	/* mask */
	.btn_level	= 15,		/* how many level can be configured. */
	.btn_min	= 0x1,		/* min value */
	.btn_max	= 0xF,		/* max value */
	.btn_inc	= 0x1,		/* increment */
};

static void __init clcd_init(void)
{
#ifdef CONFIG_FB_DOVE
	if (machine_is_titan()) {
		lcd0_dmi.clk_src = MRVL_PLL_CLK;
		lcd0_dmi.clk_name = "LCDCLK";
		lcd0_dmi.ddc_polling_disable = 1;

		lcd1_dmi.clk_src = MRVL_PLL_CLK;
		lcd1_dmi.clk_name = "LCDCLK";

		clcd_platform_init(&lcd0_dmi, &lcd0_vid_dmi,
				   &lcd1_dmi, &lcd1_vid_dmi,
				   &titan_backlight_data, NULL);
	}

	if (machine_is_mimas()) {

		u32 dev, rev;

		if (mimas_rev == MIMAS_EVT) {

			lcd0_dmi.clk_src = MRVL_PLL_CLK;
			lcd0_dmi.clk_name = "LCDCLK";

			lcd1_dmi.clk_src = MRVL_PLL_CLK;
			lcd1_dmi.clk_name = "LCDCLK";
		}

		dove_pcie_id(&dev, &rev);
		if (rev == 7) {
			/* Mimas DVT3 */
			lcd0_dmi.clk_src = MRVL_EXT_CLK1;
			lcd0_dmi.clk_name = "IDT_CLK1";
		}
		clcd_platform_init(&lcd0_dmi, &lcd0_vid_dmi,
				   &lcd1_dmi, &lcd1_vid_dmi,
				   NULL, &mimas_dcon_data);
	}
#endif /* CONFIG_FB_DOVE */
}

/*****************************************************************************
 * I2C devices:
 * 	ALC5630 codec, address 0x
 * 	Battery charger, address 0x??
 * 	G-Sensor, address 0x??
 * 	MCU PIC-16F887, address 0x??
 * 	KB3310, address 0x58
 ****************************************************************************/

static struct idt_data mimas_idt_data = {
	/* clock 0 - OUT1 connected to LCD_EXT_REF_CLK[0] */
	.clock0_enable	= 1,
	.clock0_out_id	= IDT_OUT_ID_1,
	.clock0_pll_id	= IDT_PLL_1,

	/* clock 1 - OUT2 connected to LCD_EXT_REF_CLK[1] */
	.clock1_enable	= 1,
	.clock1_out_id	= IDT_OUT_ID_2,
	.clock1_pll_id	= IDT_PLL_2,
};

static struct i2c_board_info mimas_i2c_bus0_devs[] __initdata = {
	{
		I2C_BOARD_INFO("rt5630", 0x1f),
	}, {
		I2C_BOARD_INFO("idt5v49ee503", 0x6a),
		.platform_data	= &mimas_idt_data,
	},
};

/* AT Keyboard - for Power Button event only
 * no PS/2 mouse
 */
static struct eneec_platform_data mimas_eneec_platform_data = {
	.capabilities = ENEEC_CAP_KEYBOARD,
};

static struct i2c_board_info mimas_i2c_bus1_devs[] __initdata = {
	{
		I2C_BOARD_INFO("KB39XX", 0x58),
		.irq = 7 + IRQ_DOVE_GPIO_START,
	},
};

static struct i2c_board_info mimas_evt_i2c_bus0_devs[] __initdata = {
	{
		I2C_BOARD_INFO("rt5630", 0x1f),
	},
};

static struct kb3310_platform_data mimas_evt_kb_data __initdata = {
	.enabled_ifaces = KB3310_USE_PS2_KEYBOARD | KB3310_USE_PS2_MOUSE,
};

static struct i2c_board_info mimas_evt_i2c_bus1_devs[] __initdata = {
	{
		I2C_BOARD_INFO("kb3310", 0x58),
		.irq = IRQ_DOVE_GPIO_START + 7,
		.platform_data = &mimas_evt_kb_data,
	},
};

static struct i2c_board_info titan_i2c_bus0_devs[] __initdata = {
	{
		I2C_BOARD_INFO("rt5630", 0x1f),
	},
};

static struct i2c_board_info titan_i2c_bus1_devs[] __initdata = {
	{
		I2C_BOARD_INFO("KB39XX", 0x58),
		.irq = 7 + IRQ_DOVE_GPIO_START,
	},
};

static struct spi_board_info titan_spi_info[] __initdata = {
	{
		.modalias	= "eneec_spi",
		.irq		= 7 + IRQ_DOVE_GPIO_START, //DOVE_RD_AVNG_EC_IRQ,   // irq used for EC depending on customer platform.
		.max_speed_hz	= 1*1000*1000,
		.chip_select	= 0,                     // chip-select for EC.
		.bus_num	= 1,                     // SPI bus number EC attached to.
	},
};


/*****************************************************************************
 * PCI
 ****************************************************************************/
static int __init pci_init(void)
{
	if (machine_is_mimas() || machine_is_titan())
		dove_pcie_init(1, 1);

	return 0;
}

subsys_initcall(pci_init);

/*****************************************************************************
 * Camera
 ****************************************************************************/
static struct cafe_cam_platform_data cafe_cam_data = {
	.power_down 	= 2, //CTL0 connected to the sensor power down
	.reset		= 1, //CTL1 connected to the sensor reset
	.numbered_i2c_bus = 1,
	.i2c_bus_id	= 3,
};

static int __init cam_init(void)
{
	if (machine_is_mimas() || machine_is_titan())
		dove_cam_init(&cafe_cam_data);

	return 0;
}

late_initcall(cam_init);

/*****************************************************************************
 * Audio I2S
 ****************************************************************************/
static struct orion_i2s_platform_data i2s1_data = {
	.i2s_play	= 1,
	.i2s_rec	= 1,
	.spdif_play	= 1,
};

/*****************************************************************************
 * Ethernet
 ****************************************************************************/
static struct mv643xx_eth_platform_data ge00_data = {
	.phy_addr	= 1,
};

/*****************************************************************************
 * SATA
 ****************************************************************************/
static struct mv_sata_platform_data sata_data = {
        .n_ports        = 1,
};

/*****************************************************************************
 * SPI Devices:
 *     SPI0: 4M Flash MX25L3205D
 ****************************************************************************/
static const struct flash_platform_data spi_flash_data = {
	.type           = "mx25l3205d",
};

static struct spi_board_info __initdata spi_flash_info[] = {
	{
		.modalias       = "m25p80",
		.platform_data  = &spi_flash_data,
		.irq            = -1,
		.max_speed_hz   = 20000000,
		.bus_num        = 0,
		.chip_select    = 0,
	},
};

/*****************************************************************************
 * ENE EC Devices:
 *     I2C1:
 ****************************************************************************/
static void __init eneec_gpio_setup(void)
{
	// set up IRQ
	set_irq_chip(gpio_to_irq(GPIO_EC_IRQ), &orion_gpio_irq_chip);
	set_irq_handler(gpio_to_irq(GPIO_EC_IRQ), handle_edge_irq);
	set_irq_type(gpio_to_irq(GPIO_EC_IRQ), IRQ_TYPE_EDGE_RISING);   // set to level low trigger
}

/*****************************************************************************
 * NAND
 ****************************************************************************/
static struct mtd_partition partition_dove[] = {
	{
		.name	= "Root",
		.offset	= 0,
		.size	= MTDPART_SIZ_FULL,
	},
};

static u64 nfc_dmamask = DMA_BIT_MASK(32);

static struct nfc_platform_data nfc_hal_data = {
	.nfc_width      = 8,
	.num_devs       = 1,
	.num_cs         = 2,
	.use_dma	= 1,
	.ecc_type	= MV_NFC_ECC_BCH_2K,
	.parts		= partition_dove,
	.nr_parts	= ARRAY_SIZE(partition_dove)
};

static struct nfc_platform_data nfc_data = {
	.nfc_width      = 8,
	.num_devs       = 1,
	.num_cs         = 2,
	.use_dma        = 1,
	.ecc_type	= MV_NFC_ECC_HAMMING,
	.parts		= partition_dove,
	.nr_parts	= ARRAY_SIZE(partition_dove)
};

static struct resource dove_nfc_resources[]  = {
	[0] = {
		.start	= (DOVE_NFC_PHYS_BASE),
		.end	= (DOVE_NFC_PHYS_BASE + 0xFF),
		.flags	= IORESOURCE_MEM,
	},
	[1] = {
		.start	= IRQ_NAND,
		.end	= IRQ_NAND,
		.flags	= IORESOURCE_IRQ,
	},
	[2] = {
		/* DATA DMA */
		.start	= 97,
		.end	= 97,
		.flags	= IORESOURCE_DMA,
	},
	[3] = {
		/* COMMAND DMA */
		.start	= 99,
		.end	= 99,
		.flags	= IORESOURCE_DMA,
	},
};

static struct platform_device dove_nfc = {
	.name		= "orion-nfc",
	.id		= -1,
	.dev		= {
		.dma_mask		= &nfc_dmamask,
		.coherent_dma_mask	= DMA_BIT_MASK(32),
		.platform_data		= &nfc_data,
	},
	.resource	= dove_nfc_resources,
	.num_resources	= ARRAY_SIZE(dove_nfc_resources),
};

static void __init nfc_init(void)
{
	/* Check if HAL driver is intended */
	if(useHalDrivers || useNandHal) {
		dove_nfc.name = "orion-nfc-hal";
		if (useNandHal && (strncmp(useNandHal, "ganged", 6) == 0)) {
			nfc_hal_data.nfc_width = 16;
			nfc_hal_data.num_devs = 2;
		}

		/* Check for ECC type directive */
		if (strcmp(useNandHal, "4bitecc") == 0) {
			nfc_hal_data.ecc_type = MV_NFC_ECC_BCH_2K;
		} else if (strcmp(useNandHal, "8bitecc") == 0) {
			nfc_hal_data.ecc_type = MV_NFC_ECC_BCH_1K;
		} else if (strcmp(useNandHal, "12bitecc") == 0) {
			nfc_hal_data.ecc_type = MV_NFC_ECC_BCH_704B;
		} else if (strcmp(useNandHal, "16bitecc") == 0) {
			nfc_hal_data.ecc_type = MV_NFC_ECC_BCH_512B;
		}

		nfc_hal_data.tclk = dove_tclk_get();
		dove_nfc.dev.platform_data = &nfc_hal_data;

	} else {  /* NON HAL driver is used */
		nfc_data.tclk = dove_tclk_get();
	}

	platform_device_register(&dove_nfc);
}

/*****************************************************************************
 * MPP
 ****************************************************************************/
static struct dove_mpp_mode common_mpp_modes[] __initdata = {
	{ 0, MPP_GPIO },	/* MCU_INTRn */
	{ 1, MPP_GPIO },	/* I2S_CODEC_IRQ */
	{ 2, MPP_PMU },		/* Standby power control */
	{ 3, MPP_PMU },		/* Power button - standby wakeup */
	{ 4, MPP_PMU },		/* M_CKE_MASK */
	{ 5, MPP_PMU },		/* DeepIdle power control */
	{ 6, MPP_PMU },		/* M_RST_MASK */
	{ 7, MPP_GPIO },	/* EC_IRQ */

	{ 8, MPP_GPIO },	/* OFF_CTRL */

	{ 9, MPP_GPIO },	/* HUB_RESETn */
	{ 10, MPP_PMU },	/* DVS SDI control */

	{ 12, MPP_SDIO1 },	/* SD1_CDn */
	{ 13, MPP_SDIO1 },	/* SD1_WP*/
	{ 14, MPP_GPIO },	/* USB_DEV_DET */
	{ 15, MPP_GPIO },	/* AU_IRQOUT */
	{ 16, MPP_GPIO },	/* PMU - DDR termination control */
	{ 17, MPP_TWSI },	/* TW_SDA Option 2 */

	{ 18, MPP_LCD },	/* LCD0_PWM */
	{ 19, MPP_TWSI },	/* TW_SCK Option 2 */
	{ 24, MPP_CAM },	/* will configure MPPs 24-39*/

	{ 45, MPP_GPIO },    	/* HEADSET_DET INT */

	{ 46, MPP_SDIO1 },	/* SD1 Group */
	{ 47, MPP_SDIO1 },	/* SD1 Group */
	{ 48, MPP_SDIO1 },	/* SD1 Group */
	{ 49, MPP_SDIO1 },	/* SD1 Group */
	{ 50, MPP_SDIO1 },	/* SD1 Group */
	{ 51, MPP_SDIO1 },	/* SD1 Group */

	{ 52, MPP_AUDIO1 },	/* AU1 Group */
	{ 53, MPP_AUDIO1 },	/* AU1 Group */
	{ 54, MPP_AUDIO1 },	/* AU1 Group */
	{ 55, MPP_AUDIO1 },	/* AU1 Group */
	{ 56, MPP_AUDIO1 },	/* AU1 Group */
	{ 57, MPP_AUDIO1 },	/* AU1 Group */

	{ 58, MPP_SPI0 },	/* will configure MPPs 58-61 */
	{ 62, MPP_UART1 },	/* UART1 active */
	{ -1 },
};

static struct dove_mpp_mode mimas_mpp_modes[] __initdata = {
	{ 11, MPP_GPIO },	/* LCM_DCM */
	{ 20, MPP_GPIO },	/* GP_WLAN_RSTn */
	{ 21, MPP_UART1 },	/* UA1_RTSn */
	{ 22, MPP_UART1 },	/* UA1_CTSn */
	{ 23, MPP_GPIO },	/* G_INT */
	{ -1 },
};

static struct dove_mpp_mode titan_mpp_modes[] __initdata = {
	{ 11, MPP_SATA_ACT },	/* SATA active indication*/
	{ 20, MPP_SPI1 },	/* SPI MISO */
	{ 21, MPP_SPI1 },	/* SPI CS */
	{ 22, MPP_SPI1 },	/* SPI MOSI */
	{ 23, MPP_SPI1 },	/* SPI SCK */
	{ 40, MPP_GPIO }, 	/* BL control for Titan only*/
	{ 42, MPP_GPIO },	/* MIC_DET INT */
	{ -1 },
};

/*****************************************************************************
 * GPIO
 ****************************************************************************/
static struct gpio common_gpios[] __initdata = {
	{
		.gpio	= GPIO_MCU_INTRn,
		.flags	= GPIOF_IN,		/* MCU interrupt */
		.label	= "MCU_INTRn",
	}, {
		.gpio	= GPIO_I2S_CODEC_IRQ,
		.flags	= GPIOF_IN,		/* Interrupt from ALC5630 */
		.label	= "I2S_CODEC_IRQ",
	}, {
		.gpio	= GPIO_MPP_DDR_TERM,
		.flags	= GPIOF_OUT_INIT_HIGH,	/* Enable DDR 1.8V */
		.label	= "MPP_DDR_TERM",
	}, {
		.gpio	= GPIO_OFF_CTRL,
		.flags	= GPIOF_OUT_INIT_HIGH,	/* Power Off */
		.label	= "OFF_CTRL",
	}, {
		.gpio	= GPIO_EC_IRQ,
		.flags	= GPIOF_IN,		/* EC Interrupt */
		.label	= "EC_TO_DOVE_IRQ",
	}, {
		.gpio	= GPIO_HUB_RESETn,
		.flags	= GPIOF_OUT_INIT_HIGH,	/* HUB_RESETn */
		.label	= "HUB_RESETn",
	}, {
		.gpio	= GPIO_LCM_DCM,
		.flags	= GPIOF_OUT_INIT_HIGH,	/* Enable LCD Power */
		.label	= "LCM_DCM",
	}, {
		.gpio	= GPIO_HEADSET_DET,
		.flags	= GPIOF_IN,		/* Interrupt from audio codec */
		.label	= "HEADSET_DET",
	},
};

static struct gpio mimas_gpios[] __initdata = {
	{
		.gpio	= GPIO_GP_WLAN_RSTn,
		.flags	= GPIOF_OUT_INIT_HIGH,
		.label	= "GP_WLAN_RSTn",
	}, {
		.gpio	= GPIO_G_INT,
		.flags	= GPIOF_IN,		/* Interrupt from G-sensor */
		.label	= "G_INT",
	},
};

static struct gpio titan_gpios[] __initdata = {
	{
		.gpio	= GPIO_TITAN_LCM_BL_CTRL,
		.flags	= GPIOF_OUT_INIT_HIGH,	/* Enable LCD backlight */
		.label	= "LCM_BL_CTRL",
	}, {
		.gpio	= GPIO_TITAN_MIC_DET,
		.flags	= GPIOF_IN,		/* Interrupt from audio codec */
		.label	= "MIC_DET",
	},
};

static void __init gpio_init(void)
{
	struct gpio *p = &common_gpios[0];
	int i, err = 0;

	for (i = 0; i < ARRAY_SIZE(common_gpios); i++, p++) {
		orion_gpio_set_valid(p->gpio, 1);
		err = gpio_request_one(p->gpio, p->flags, p->label);
		if (err)
			pr_err("failed to setup GPIO for %s\n", p->label);
	}

	if (machine_is_mimas()) {
		p = &mimas_gpios[0];
		for (i = 0; i < ARRAY_SIZE(mimas_gpios); i++, p++) {
			orion_gpio_set_valid(p->gpio, 1);
			err = gpio_request_one(p->gpio, p->flags, p->label);
			if (err)
				pr_err("failed to setup GPIO for %s\n", p->label);
		}
	}

	if (machine_is_titan()) {
		p = &titan_gpios[0];
		for (i = 0; i < ARRAY_SIZE(titan_gpios); i++, p++) {
			orion_gpio_set_valid(p->gpio, 1);
			err = gpio_request_one(p->gpio, p->flags, p->label);
			if (err)
				pr_err("failed to setup GPIO for %s\n", p->label);
		}
	}
}

#ifdef CONFIG_PM
extern int global_dvs_enable;
/*****************************************************************************
 * POWER MANAGEMENT
 ****************************************************************************/
static int __init pm_init(void)
{
	MV_PMU_INFO pmuInitInfo;
	u32 dev, rev;

	if (!(machine_is_mimas() || machine_is_titan()))
		return 0;

	global_dvs_enable = dvs_enable;

	dove_pcie_id(&dev, &rev);

	pmuInitInfo.batFltMngDis = MV_FALSE;			/* Keep battery fault enabled */
	pmuInitInfo.exitOnBatFltDis = MV_FALSE;			/* Keep exit from STANDBY on battery fail enabled */
	pmuInitInfo.sigSelctor[0] = PMU_SIGNAL_NC;
	pmuInitInfo.sigSelctor[1] = PMU_SIGNAL_NC;
	pmuInitInfo.sigSelctor[2] = PMU_SIGNAL_SLP_PWRDWN;	/* STANDBY => 0: I/O off, 1: I/O on */
	pmuInitInfo.sigSelctor[3] = PMU_SIGNAL_EXT0_WKUP;	/* power on push button */
	if (rev >= DOVE_REV_X0) { /* For X0 and higher Power Good indication is not needed */
		if (standby_fix)
			pmuInitInfo.sigSelctor[4] = PMU_SIGNAL_CKE_OVRID;	/* CKE controlled by Dove */
		else
			pmuInitInfo.sigSelctor[4] = PMU_SIGNAL_NC;
	}
	else
		pmuInitInfo.sigSelctor[4] = PMU_SIGNAL_CPU_PWRGOOD;	/* CORE power good used as Standby PG */

	pmuInitInfo.sigSelctor[5] = PMU_SIGNAL_CPU_PWRDWN;	/* DEEP-IdLE => 0: CPU off, 1: CPU on */

	if ((rev >= DOVE_REV_X0) && (standby_fix)) /* For boards with X0 we use MPP6 as MRESET */
		pmuInitInfo.sigSelctor[6] = PMU_SIGNAL_MRESET_OVRID;		/* M_RESET is pulled up - always HI */
	else
		pmuInitInfo.sigSelctor[6] = PMU_SIGNAL_NC;
	pmuInitInfo.sigSelctor[7] = PMU_SIGNAL_1;		/* Standby Led - inverted */
	pmuInitInfo.sigSelctor[8] = PMU_SIGNAL_NC;
	pmuInitInfo.sigSelctor[9] = PMU_SIGNAL_NC;		/* CPU power good  - not used */
	pmuInitInfo.sigSelctor[10] = PMU_SIGNAL_SDI;		/* Voltage regulator control */
	pmuInitInfo.sigSelctor[11] = PMU_SIGNAL_NC;
	pmuInitInfo.sigSelctor[12] = PMU_SIGNAL_NC;
	pmuInitInfo.sigSelctor[13] = PMU_SIGNAL_NC;
	pmuInitInfo.sigSelctor[14] = PMU_SIGNAL_NC;
	pmuInitInfo.sigSelctor[15] = PMU_SIGNAL_NC;
	pmuInitInfo.dvsDelay = 0x4200;				/* ~100us in 166MHz cc - delay for DVS change */
	if (rev >= DOVE_REV_X0) { /* For X0 and higher wait at least 150ms + spare */
		pmuInitInfo.standbyPwrDelay = 0x2000;		/* 250ms delay to wait for complete powerup */
		pmuInitInfo.ddrTermGpioNum = 16;		/* GPIO 16 used to disable terminations */
	} else {
		pmuInitInfo.standbyPwrDelay = 0x140;		/* 10ms delay after getting the power good indication */
		pmuInitInfo.ddrTermGpioNum = 6;			/* GPIO 6 used to disable terminations */
	}

	/* Initialize the PMU HAL */
	if (mvPmuInit(&pmuInitInfo) != MV_OK)
	{
		printk(KERN_ERR "ERROR: Failed to initialise the PMU!\n");
		return 0;
	}

	/* Configure wakeup events */
	mvPmuWakeupEventSet(PMU_STBY_WKUP_CTRL_EXT0_FALL | PMU_STBY_WKUP_CTRL_RTC_MASK);

	/* Register the PM operation in the Linux stack */
	dove_pm_register();

	return 0;
}

__initcall(pm_init);
#endif

#ifdef CONFIG_BATTERY_MCU
void __init dove_battery_init(void)
{
	platform_device_register_simple("battery", 0, NULL, 0);
}
#endif

#ifdef CONFIG_ANDROID_PMEM
#include <linux/dma-mapping.h>
#include <linux/android_pmem.h>

void android_add_pmem(char *name, size_t size, int no_allocator, int cached)
{
        struct platform_device *android_pmem_device;
        struct android_pmem_platform_data *android_pmem_pdata;
        struct page *page;
        unsigned long addr, tmp;
        static int id;
        unsigned long paddr = 0;

        android_pmem_device = kzalloc(sizeof(struct platform_device), GFP_KERNEL);
        if(android_pmem_device == NULL)
                return ;

        android_pmem_pdata = kzalloc(sizeof(struct android_pmem_platform_data), GFP_KERNEL);
        if(android_pmem_pdata == NULL) {
                kfree(android_pmem_device);
                return ;
        }

        page = alloc_pages(GFP_KERNEL, get_order(size));
        if (page == NULL)
                return ;

        addr = (unsigned long)page_address(page);
        paddr = virt_to_phys((void *)addr);
        tmp = size;
        dma_cache_maint(addr, size, DMA_FROM_DEVICE);
        while(tmp > 0) {
                SetPageReserved(virt_to_page(addr));
                addr += PAGE_SIZE;
                tmp -= PAGE_SIZE;
        }
        android_pmem_pdata->name = name;
        android_pmem_pdata->start = paddr;
        android_pmem_pdata->size = size;
        android_pmem_pdata->no_allocator = no_allocator ;
        android_pmem_pdata->cached = cached;

        android_pmem_device->name = "android_pmem";
        android_pmem_device->id = id++;
        android_pmem_device->dev.platform_data = android_pmem_pdata;

        platform_device_register(android_pmem_device);
}
#endif

static void power_off(void)
{
	gpio_set_value(GPIO_OFF_CTRL, 0);
}

/*****************************************************************************
 * Board Init
 ****************************************************************************/
static void __init common_init(void)
{
	u32 dev, rev;

	/*
	 * Basic Dove setup. Needs to be called early.
	 */
	dove_init();
	dove_pcie_id(&dev, &rev);

	/* Initialize gpiolib */
	orion_gpio_init();

	/* decide Mimas board revision */
	if (machine_is_mimas()) {
		mimas_rev = rev < DOVE_REV_A0 ? MIMAS_EVT : MIMAS_DVT;
		printk("Mimas %s board detected\n", (mimas_rev == MIMAS_EVT) ?
				"EVT" : "DVT");
	}

	if (machine_is_titan()) {
		titan_rev = rev < DOVE_REV_A0 ? TITAN_EVT : TITAN_DVT;
		printk("Titan %s board detected\n", (titan_rev == MIMAS_EVT) ?
				"EVT" : "DVT");
	}

	dove_mpp_conf(common_mpp_modes);

	if (machine_is_mimas())
		dove_mpp_conf(mimas_mpp_modes);

	if (machine_is_titan())
		dove_mpp_conf(titan_mpp_modes);

	gpio_init();

	pm_power_off = power_off;

	/* sdio card interrupt workaround using GPIOs */
	dove_sd_card_int_wa_setup(1);

	/* Set the EC gpio and irq. */
	eneec_gpio_setup();

	dove_rtc_init();
	pxa_init_dma_wins(&dove_mbus_dram_info);
	pxa_init_dma(16);

	if (useHalDrivers || useNandHal) {
		if (mvPdmaHalInit(MV_PDMA_MAX_CHANNELS_NUM) != MV_OK) {
			printk(KERN_ERR "mvPdmaHalInit() failed.\n");
			BUG();
		}
		/* reserve channels for NAND Data and command PDMA */
		pxa_reserve_dma_channel(MV_PDMA_NAND_DATA);
		pxa_reserve_dma_channel(MV_PDMA_NAND_COMMAND);
	}

	dove_xor0_init();
	dove_xor1_init();
#ifdef CONFIG_MV_ETHERNET
	if (use_hal_giga || useHalDrivers)
		dove_mv_eth_init();
	else
#endif
	dove_ge00_init(&ge00_data);
	dove_ehci0_init();
	dove_ehci1_init();
	/* ehci init functions access the usb port, only now it's safe to disable
	 * all clocks
	 */
	ds_clks_disable_all(0, 0);
	dove_sata_init(&sata_data);

	if (machine_is_titan() && titan_rev >= TITAN_DVT) {
		spi_register_board_info(titan_spi_info, ARRAY_SIZE(titan_spi_info));
		dove_spi1_init(0);
	}

	/* uart0 is the debug port, register it first so it will be */
	/* represented by device ttyS0, root filesystems usually expect the */
	/* console to be on that device */
	dove_uart0_init();
	dove_uart1_init();
	dove_i2c_init();
	dove_i2c_exp_init(0);
	if (rev >= DOVE_REV_X0) {
		dove_i2c_exp_init(1);
	}
	dove_sdhci_cam_mbus_init();
	dove_sdio1_init();
	nfc_init();
	clcd_init();
	dove_vmeta_init();
	dove_gpu_init();
	dove_cesa_init();
	dove_hwmon_init();

	dove_i2s_init(1, &i2s1_data);

	if (machine_is_mimas()) {
		if (mimas_rev == MIMAS_EVT) {
			i2c_register_board_info(0, mimas_evt_i2c_bus0_devs,
					ARRAY_SIZE(mimas_evt_i2c_bus0_devs));
			i2c_register_board_info(1, mimas_evt_i2c_bus1_devs,
					ARRAY_SIZE(mimas_evt_i2c_bus1_devs));
		} else {
			i2c_register_board_info(0, mimas_i2c_bus0_devs,
					ARRAY_SIZE(mimas_i2c_bus0_devs));
			i2c_register_board_info(1, mimas_i2c_bus1_devs,
					ARRAY_SIZE(mimas_i2c_bus1_devs));
		}
	}

	if (machine_is_titan()) {
		i2c_register_board_info(0, titan_i2c_bus0_devs,
				ARRAY_SIZE(titan_i2c_bus0_devs));
		i2c_register_board_info(1, titan_i2c_bus1_devs,
				ARRAY_SIZE(titan_i2c_bus1_devs));
	}

	spi_register_board_info(spi_flash_info, ARRAY_SIZE(spi_flash_info));
#ifdef CONFIG_BATTERY_MCU
	dove_battery_init();
#endif
#ifdef CONFIG_ANDROID_PMEM
        android_add_pmem("pmem", 0x01800000UL, 1, 0);
        android_add_pmem("pmem_adsp", 0x04000000UL, 0, 0);
#endif
}

MACHINE_START(MIMAS, "WYSE Mimas Board")
	.phys_io	= DOVE_SB_REGS_PHYS_BASE,
	.io_pg_offst	= ((DOVE_SB_REGS_VIRT_BASE) >> 18) & 0xfffc,
	.boot_params	= 0x00000100,
	.init_machine	= common_init,
	.map_io		= dove_map_io,
	.init_irq	= dove_init_irq,
	.timer		= &dove_timer,
	/* reserve memory for VMETA and GPU */
	.fixup		= dove_tag_fixup_mem32,	/* reserve memory for VMETA/GPU */
MACHINE_END

MACHINE_START(TITAN, "WYSE Titan Board")
	.phys_io	= DOVE_SB_REGS_PHYS_BASE,
	.io_pg_offst	= ((DOVE_SB_REGS_VIRT_BASE) >> 18) & 0xfffc,
	.boot_params	= 0x00000100,
	.init_machine	= common_init,
	.map_io		= dove_map_io,
	.init_irq	= dove_init_irq,
	.timer		= &dove_timer,
	.fixup		= dove_tag_fixup_mem32,	/* reserve memory for VMETA/GPU */
MACHINE_END

static void __init early_setup_mimas(char **p)
{
	__machine_arch_type = MACH_TYPE_MIMAS;
}

__early_param("mimas", early_setup_mimas);

static void __init early_setup_titan(char **p)
{
	__machine_arch_type = MACH_TYPE_TITAN;
}

__early_param("titan", early_setup_titan);
