/*
 * linux/drivers/input/serio/kb3310.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License.
 *
 */
#include <linux/module.h>
#include <linux/device.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/input.h>
#include <linux/i2c.h>
#include <linux/init.h>
#include <linux/serio.h>
#include <linux/delay.h>

#ifdef CONFIG_MACH_MIMAS
#include <linux/kernel.h>
#include <linux/string.h>
#define DEBUG_UDEV
#ifdef DEBUG_UDEV
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/uaccess.h>
#endif
#endif

#include <linux/i2c/kb3310.h>


/* #define KBC_DEBUG */

#define KB3310_MAX_DATA		2

#define KB3310_RESET_CMD	0x99
#define PORT_ECRAM 		0x80
/* Define the used RAM register address for reading content purpose */
#ifdef DEBUG_UDEV
#define MAX_ECRAM_REG		4
static const u8 ram_reg[] = { 0x01, 0x02, 0x03, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x20, 0x21, 0x22 ,0x30 ,0x31 };
#endif

/*
 * KB3310 data read byte 0 fields
 */
#define KB3310_READ_TOGGLE	0x01
#define KB3310_READ_DATA	0x02

struct kb3310_data {
	struct mutex		lock;
	struct i2c_client	*client;
	struct work_struct	work;
	unsigned int		pm_suspend;
	unsigned int		enabled_ifaces;
	struct serio		*keybd_io;
	struct serio		*mouse_io;
	int			ref_count;
	unsigned int		num_of_retries;
	unsigned int		retry_delay;
	unsigned char		gpio_devid;
	unsigned char		keybd_devid;
	unsigned char		mouse_devid;
	unsigned char		smbus_devid;
#ifdef CONFIG_MACH_MIMAS
	unsigned char		ecram_devid;
#endif
	int			toggle;
};

static struct workqueue_struct	*kb3310_wq;

#ifdef DEBUG_UDEV
static int enedev_devs = 1;
static int enedev_major = 0;
static int enedev_minor = 0;
static struct cdev enedev_cdev;
static struct class *enedev_class = NULL;
static dev_t enedev_dev;
static char val[256];
static struct i2c_client *ene_client;
static int ene_i2c_write(struct kb3310_data *kb, u8 cmd, u8 *buf, int len);
static int ene_i2c_read(struct kb3310_data *kb, u8 cmd, u8 *buf, int len);


/* user can read RAM register content from user space by using cat command.
 * This function will only read data from val array which was updated when 
 * the module has been installed or any update to the EC RAM. 
 */
ssize_t enedev_read(struct file *file, char __user *buf, size_t count, loff_t *f_pos)
{
	int retval;
	int i;
	char *temp = kmalloc(PAGE_SIZE, GFP_KERNEL);
	u16 data_1;
	
	printk("Reg:Data\n");
	count = sprintf(temp, "Reg:Data\n");
	for(i=0;i<sizeof(ram_reg);i++){
                if(ram_reg[i]==0x22){
			char buff[3];
                        retval = i2c_smbus_write_word_data(ene_client, 0x80|0x01, 0x22);
                        data_1=i2c_smbus_read_word_data(ene_client, 0x00);
			sprintf(buff,"%02x",data_1>>8);
			memcpy(&val[0x22*2],buff,2);
                }
		printk("%02x:%c%c\n",ram_reg[i],val[ram_reg[i]*2],val[ram_reg[i]*2+1]);
		count += sprintf(temp + count,"%02x:%c%c\n",ram_reg[i],val[ram_reg[i]*2],val[ram_reg[i]*2+1]);
	}

	printk("read: fpos=%d count=%d\n", (int)*f_pos, count);

	if(copy_to_user(buf, temp, count)) {
		retval = -EFAULT;
		goto out;
	}

	if(*f_pos >= count)
		retval = 0;
	else{
		*f_pos += count;
		retval = count;
	}
out:
	kfree(temp);
	return (retval);
}

struct file_operations enedev_fops = 
{
	.read = enedev_read,
};
#endif

/*
 * To write, we just access the chip's address in write mode, and dump the
 * command and data out on the bus.  The data are taken as sequential u8s,
 * to a maximum of KB3310_MAX_DATA.
 */
static int ene_i2c_write(struct kb3310_data *kb, u8 cmd, u8 *buf, int len)
{
	int i, ret = 0;
	u8 data[KB3310_MAX_DATA+1];

	if (unlikely(len > KB3310_MAX_DATA)) {
		dev_err(&kb->client->dev, "tried to send %d bytes\n", len);
		return 0;
	}

	data[0] = cmd;
	for (i = 0; i < len; i++)
		data[i+1] = buf[i];
	for ( ; i < KB3310_MAX_DATA; i++)
		data[i+1] = 0;

	mutex_lock(&kb->lock);

	/*
	 * If the EC is busy (pulling battery status on another i2c)
	 * while we send the data, we can get a timeout,
	 * so try a few times with a delay.
	 */
	for (i = kb->num_of_retries; i > 0; i--) {
		ret = i2c_master_send(kb->client, data, KB3310_MAX_DATA+1);
		if (unlikely(ret == -ETIMEDOUT)) {
			mdelay(kb->retry_delay);
			continue;
		}
		if (ret > 0)
			break;
	}

	mutex_unlock(&kb->lock);

	if (unlikely(ret != (KB3310_MAX_DATA+1)))
		dev_err(&kb->client->dev, "sent %d bytes of %d total\n",
			(KB3310_MAX_DATA+1), ret);

	return len;
}

/*
 * To read, we first send the command byte to the chip and end the transaction,
 * then access the chip in read mode, at which point it will send the data.
 */
static int ene_i2c_read(struct kb3310_data *kb, u8 cmd, u8 *buf, int len)
{
	int i, ret = 0;

	mutex_lock(&kb->lock);

	/*
	 * If the EC is busy (pulling battery status on another i2c)
	 * while we send the data, we can get a timeout,
	 * so try a few times with a delay.
	 */
	for (i = kb->num_of_retries; i > 0; i--) {
		ret = i2c_master_send(kb->client, &cmd, 1);
		if (unlikely(ret == -ETIMEDOUT)) {
			mdelay(kb->retry_delay);
			continue;
		}
		if (ret > 0)
			break;
	}

	if (unlikely(ret != 1))
		dev_err(&kb->client->dev, "sending read cmd 0x%02x failed\n",
			cmd);

	ret = i2c_master_recv(kb->client, buf, len);

	mutex_unlock(&kb->lock);

	if (unlikely(ret != len))
		dev_err(&kb->client->dev, "wanted %d bytes, got %d\n",
			len, ret);

#ifdef KBC_DEBUG
	printk("R:%02x %02x\n", buf[0], buf[1]);
#endif

	return ret;
}

static int kb3310_write(struct serio *io, unsigned char val)
{
	struct kb3310_data *kb = (struct kb3310_data *)io->port_data;
	u8 cmd, buf[KB3310_MAX_DATA];
	int ret;

	if (io == kb->keybd_io)
		cmd = kb->keybd_devid;
	else if (io == kb->mouse_io)
		cmd = kb->mouse_devid;
	else {
		dev_err(&kb->client->dev,
			"write reguest from unregistered interface\n");
		return 0;
	}

	buf[0] = val;
	buf[1] = 0;

#ifdef KBC_DEBUG
	printk("W:%02x %02x %02x\n", cmd, buf[0], buf[1]);
#endif

	ret = ene_i2c_write(kb, cmd, buf, KB3310_MAX_DATA);

	return (ret == KB3310_MAX_DATA) ? 0 : SERIO_TIMEOUT;
}

#ifdef CONFIG_MACH_MIMAS
#if 0
static ssize_t ec_ram_read(struct kobject *kobj, struct bin_attribute *attr, char *buf, loff_t off, size_t count)
{
	u8 cmd = 0;
	int ret;
	u16 i;
	struct i2c_client *client = to_i2c_client(container_of(kobj, struct device, kobj));
	struct kb3310_data *kb = i2c_get_clientdata(client);
	u8 buffer[2]="";


	/* Read ECRAM from register 0x01 to 0x03 and then display the read value on console *
	 * cmd = ECRAM device ID | KB3310_READ_TOGGLE = 0x81				    *
	 * send buffer[0] = Register							    *
	 * send buffer[1] = 0x00							    *
	 * return buf[0] = ECRAM device ID						    *
	 * return buf[1] = Value in Register						    */ 
	for(i=1;i<4;i++){
		cmd = kb->ecram_devid|KB3310_READ_TOGGLE;
		buffer[0] = i; 
		buffer[1] = 0x00;
		ret = ene_i2c_write(kb, cmd, buffer, KB3310_MAX_DATA);
		printk("kb3310 Read_Write: id:%02x reg:%02x data:%02x\n", cmd, buffer[0], buffer[1]);
		cmd = 0;
		ret = ene_i2c_read(kb, cmd, buf, KB3310_MAX_DATA);
		//serio_interrupt(kb->keybd_io, buf[1]+1, 0);	//convert buf[1] to scancode and display it on console
		//serio_interrupt(kb->keybd_io, (buf[1]+1)|0x80, 0);	//send a break scancode
		printk("kb3310 Read_Resp: id:%02x reg:%02x data:%02x\n", buf[0], buffer[0], buf[1]);
	}

	return 0;
}
#endif

/* User can use ec_ram_write function to set ram value to its RAM register from user space. */
static ssize_t ec_ram_write(struct kobject *kobj, struct bin_attribute *bin_attr, char *buf, loff_t off, size_t count)
{
	u8 cmd = 0;
	int ret, i;
	struct i2c_client *client = to_i2c_client(container_of(kobj, struct device, kobj));
	struct kb3310_data *kb = i2c_get_clientdata(client);
	u8 buffer[2]="", *temp;

	/* Write buffer value to ECRAM  *
 	 * cmd = ECRAM device ID = 0x80	*
	 * buffer[0] = Register		*
	 * buffer[1] = data		*/	
	if(count>3){
		cmd = kb->ecram_devid;
		buffer[0] = (u8) simple_strtoul(buf,NULL,16);
		temp = strchr(buf, 0x20); //string with space character 
		buffer[1] = (u8) simple_strtoul(&temp[1],NULL,16);
		ret = ene_i2c_write(kb, cmd, buffer, KB3310_MAX_DATA);
		printk("kb3310 Write: ID:%02x Reg:%02x Dat:%02x\n", cmd, buffer[0], buffer[1]);
	}

#ifdef DEBUG_UDEV
                /* Update val table by reading ECRAM registers */
                for(i=0;i<sizeof(ram_reg);i++){
			if(ram_reg[i]!=buffer[0]) continue;
			else{
				char buff[3];
				u16 data_1;
                        	data_1 = ram_reg[i] | (((u16) 0x0)<<8);
                        	mdelay(50);
                        	ret = i2c_smbus_write_word_data(ene_client, PORT_ECRAM|0x01, data_1);
                        	mdelay(50);
                        	data_1 = i2c_smbus_read_word_data(ene_client, PORT_ECRAM);
                        	/*printk("kb3310: R:%04x\n", data_1);*/
				sprintf(buff, "%02x", (data_1>>8));
				memcpy(&val[ram_reg[i]*2],buff,2);
			}
               }
#endif
	return count;
}

static struct bin_attribute ec_ver_attr = {	
	.attr = {
		.name = "ec_ram",
//		.mode = S_IRUGO | S_IWUGO,
		.mode = S_IWUGO,
		.owner = THIS_MODULE,
	},
	.size = 256,
	.write = ec_ram_write,
//	.read = ec_ram_read,
};
#endif

/*
 * Bottom half: handle the interrupt by posting key events, or dealing with
 * errors appropriately.
 */
static void kb3310_work(struct work_struct *work)
{
	struct kb3310_data *kb = container_of(work, struct kb3310_data, work);
	u8 cmd = 0;
	u8 buf[KB3310_MAX_DATA];
	int ret, toggle;

	ret = ene_i2c_read(kb, cmd, buf, KB3310_MAX_DATA);

	toggle = (int)(buf[0] & KB3310_READ_TOGGLE);

	if (unlikely((kb->toggle ^ toggle) == 0)) {
		dev_warn(&kb->client->dev, "invalid toggle bit\n");
	} else {
		if ((kb->enabled_ifaces & KB3310_USE_PS2_KEYBOARD) &&
			((buf[0] & kb->keybd_devid) == kb->keybd_devid)) {
			dev_dbg(&kb->client->dev, "keybd: %s %x\n",
				(buf[0] & KB3310_READ_DATA) ? "data" : "rsp",
				(unsigned int)buf[1]);
			serio_interrupt(kb->keybd_io, buf[1], 0);
		}

		if ((kb->enabled_ifaces & KB3310_USE_PS2_MOUSE) &&
			((buf[0] & kb->mouse_devid) == kb->mouse_devid)) {
			dev_dbg(&kb->client->dev, "mouse: %s %x\n",
				(buf[0] & KB3310_READ_DATA) ? "data" : "rsp",
				(unsigned int)buf[1]);
			serio_interrupt(kb->mouse_io, buf[1], 0);
		}

		kb->toggle = toggle;
	}

	enable_irq(kb->client->irq);
}

/*
 * We cannot use I2C in interrupt context, so we just schedule work.
 */
static irqreturn_t kb3310_irq(int irq, void *data)
{
	struct kb3310_data *kb = (struct kb3310_data *)data;

	disable_irq_nosync(irq);
	queue_work(kb3310_wq, &kb->work);

#ifdef KBC_DEBUG
	printk("int\n");
#endif

	return IRQ_HANDLED;
}

/*
 * Read the chip ID.
 */
static int kb3310_read_id(struct kb3310_data *kb, u8 *buf)
{
	return 0;
}

static int kb3310_open(struct serio *io)
{
	struct kb3310_data *kb = (struct kb3310_data *)io->port_data;
	int ret;

#ifdef KBC_DEBUG
	printk("kb3310_open: io=%08x, ref=%d\n", (u32)io, kb->ref_count);
#endif
	if ((kb->enabled_ifaces & KB3310_USE_PS2_KEYBOARD) ||
		(kb->enabled_ifaces & KB3310_USE_PS2_MOUSE)) {
		if (! kb->ref_count) {
			INIT_WORK(&kb->work, kb3310_work);

			ret = request_irq(kb->client->irq, kb3310_irq, 0,
					"kb3310", (void *)kb);
			if (ret) {
				dev_err(&kb->client->dev,
					"could not get keyboard/mouse IRQ %d\n",
					kb->client->irq);
				return ret;
			}
#if 0
			/* Configure for rising edge trigger */
			set_irq_chip(kb->client->irq, &orion_gpio_irq_chip);
			set_irq_handler(kb->client->irq, handle_edge_irq);
			set_irq_type(kb->client->irq, IRQ_TYPE_EDGE_RISING);
#else
			/* Configure for high level trigger */
			set_irq_type(kb->client->irq, IRQ_TYPE_LEVEL_HIGH);
#endif
		}

		mutex_lock(&kb->lock);
		kb->ref_count++;
		mutex_unlock(&kb->lock);
	}

	return 0;
}

static void kb3310_close(struct serio *io)
{
	struct kb3310_data *kb = (struct kb3310_data *)io->port_data;

#ifdef KBC_DEBUG
	printk("kb3310_close: io=%08x, ref=%d\n", (u32)io, kb->ref_count);
#endif

	if ((kb->enabled_ifaces & KB3310_USE_PS2_KEYBOARD) ||
		(kb->enabled_ifaces & KB3310_USE_PS2_MOUSE)) {
		mutex_lock(&kb->lock);
		kb->ref_count--;
		mutex_unlock(&kb->lock);

		if (! kb->ref_count) {
			free_irq(kb->client->irq, (void *)kb);
			cancel_work_sync(&kb->work);
			flush_workqueue(kb3310_wq);
		}
	}
}

static int __devinit kb3310_probe(struct i2c_client *client,
				const struct i2c_device_id *id)
{
	struct kb3310_data *kb;
	struct kb3310_platform_data *pdata;
	struct serio *keybd_io = NULL, *mouse_io = NULL;
	u8 cmd = 0, buf[KB3310_MAX_DATA];
	u16 data = 0, tmp;
	int i, ret;

#ifdef DEBUG_UDEV
	dev_t dev = MKDEV(enedev_major, 0);
	int alloc_ret = 0;
	int major;
	int cdev_err = 0;
	struct device *class_dev = NULL;
#endif

	kb = kzalloc(sizeof(struct kb3310_data), GFP_KERNEL);
	if (!kb) {
		ret = -ENOMEM;
		goto release1;
	}

	mutex_init(&kb->lock);

	pdata			= client->dev.platform_data;
	kb->enabled_ifaces	= pdata->enabled_ifaces;
	kb->num_of_retries	= 5;
	kb->retry_delay		= 1;		/* 1 ms */
	kb->gpio_devid		= 0x20;
	kb->keybd_devid		= 0x30;
	kb->mouse_devid		= 0x40;
	kb->smbus_devid		= 0x50;
#ifdef CONFIG_MACH_MIMAS
	kb->ecram_devid		= 0x80;
#endif

	i2c_set_clientdata(client, kb);
	kb->client = client;
#ifdef DEBUG_UDEV
	ene_client = client;
#endif
	if (kb3310_read_id(kb, buf) != 0) {
		dev_err(&client->dev, "device not found\n");
		ret = -ENODEV;
		goto release2;
	}

	buf[0] = buf[1] = 0;

	/*
	 * Drain old data out of device and we do not care whether the data
	 * came from keyboard or mouse.
	 */
	ret = ene_i2c_read(kb, cmd, buf, KB3310_MAX_DATA);
	data = (u16) buf[1];
	data <<= 8;
	data |= (u16) buf[0];

	dev_dbg(&client->dev, "drain data : 0x%04x\n", (unsigned int)data);

	for (i = 32; i > 0; i--) {
		ret = ene_i2c_read(kb, cmd, buf, KB3310_MAX_DATA);
		tmp = (u16) buf[1];
		tmp <<= 8;
		tmp |= (u16) buf[0];

		/* no new data */
		if (tmp == data)
			break;
		else {
			data = tmp;
			dev_dbg(&client->dev, "drain data : 0x%04x\n",
				(unsigned int)data);
		}
	}

	kb->toggle = (int)(data & KB3310_READ_TOGGLE);

	buf[0] = KB3310_RESET_CMD;
	buf[1] = 0;

	if (kb->enabled_ifaces & KB3310_USE_PS2_KEYBOARD) {
		keybd_io = kzalloc(sizeof(struct serio), GFP_KERNEL);
		if (!keybd_io) {
			ret = -ENOMEM;
			goto release2;
		}

		cmd = kb->keybd_devid;
		ret = ene_i2c_write(kb, cmd, buf, KB3310_MAX_DATA);
#ifdef KBC_DEBUG
		printk("W:%02x %02x %02x\n", cmd, buf[0], buf[1]);
#endif

		keybd_io->id.type	= SERIO_8042_XL;
		keybd_io->write		= kb3310_write;
		keybd_io->open		= kb3310_open;
		keybd_io->close		= kb3310_close;
		snprintf(keybd_io->name, sizeof(keybd_io->name),
			"%s ps/2 keyboard", client->driver->driver.name);
		snprintf(keybd_io->phys, sizeof(keybd_io->phys),
			"i2c/addr-%x/dev-%x/irq-%x", client->addr,
			kb->keybd_devid, client->irq);
		keybd_io->port_data	= kb;
		keybd_io->dev.parent	= &client->dev;
		kb->keybd_io		= keybd_io;

		serio_register_port(kb->keybd_io);
	}

	if (kb->enabled_ifaces & KB3310_USE_PS2_MOUSE) {
		mouse_io = kzalloc(sizeof(struct serio), GFP_KERNEL);
		if (!mouse_io) {
			ret = -ENOMEM;
			goto release3;
		}

		cmd = kb->mouse_devid;
		ret = ene_i2c_write(kb, cmd, buf, KB3310_MAX_DATA);
#ifdef KBC_DEBUG
		printk("W:%02x %02x %02x\n", cmd, buf[0], buf[1]);
#endif

		mouse_io->id.type	= SERIO_8042;
		mouse_io->write		= kb3310_write;
		mouse_io->open		= kb3310_open;
		mouse_io->close		= kb3310_close;
		snprintf(mouse_io->name, sizeof(mouse_io->name),
			"%s ps/2 mouse", client->driver->driver.name);
		snprintf(mouse_io->phys, sizeof(mouse_io->phys),
			"i2c/addr-%x/dev-%x/irq-%x", client->addr,
			kb->mouse_devid, client->irq);
		mouse_io->port_data	= kb;
		mouse_io->dev.parent	= &client->dev;
		kb->mouse_io		= mouse_io;

		serio_register_port(kb->mouse_io);
	}

	dev_info(&client->dev, "KB3310 PS2 Keyboard/Mouse driver loaded!\n");

#ifdef CONFIG_MACH_MIMAS
	/* create the sysfs file for ec_ram_write */
	ret = sysfs_create_bin_file(&client->dev.kobj, &ec_ver_attr);
#ifdef DEBUG_UDEV
	alloc_ret = alloc_chrdev_region(&dev, 0, enedev_devs, "enedev");
	if(alloc_ret)
		goto error;
	enedev_major = major = MAJOR(dev);
	
	cdev_init(&enedev_cdev, &enedev_fops);
	enedev_cdev.owner = THIS_MODULE;
	enedev_cdev.ops = &enedev_fops;
	cdev_err = cdev_add(&enedev_cdev, MKDEV(enedev_major, enedev_minor), 1);
	if (cdev_err)
		goto error;
	/* register class */
	enedev_class = class_create(THIS_MODULE, "enedev");
	if (IS_ERR(enedev_class)){
		goto error;
	}
	enedev_dev = MKDEV(enedev_major, enedev_minor);
	class_dev = device_create(enedev_class, NULL, enedev_dev, NULL, "enedev%d", enedev_minor);
	printk(KERN_INFO "enedev driver(major %d) installed.\n", major);

        /* Read ECRAM from register */
        tmp=0;
        for(i=0;i<sizeof(ram_reg);i++){
		u16 data_1;
                data_1 = ram_reg[i] | (((u16) 0x0) << 8);
                ret  = i2c_smbus_write_word_data(client, PORT_ECRAM|0x01, data_1);
                /*printk("kb3310 Read_Write: id:%02x reg:%02x \n", (PORT_ECRAM|0x01), ram_reg[i] );*/
                data_1 = i2c_smbus_read_word_data(client, 0);
                printk("kb3310 Read_Resp: id:%02x reg:%02x data:%02x\n", (data_1&0xFF), ram_reg[i], (data_1>>8));
                tmp += sprintf(&val[ram_reg[i]*2], "%02x", (data_1>>8));
        }
#endif

#endif /* CONFIG_MACH_MIMAS */

	return 0;

#ifdef DEBUG_UDEV
error:
	if (cdev_err == 0){
		cdev_del(&enedev_cdev);
        	ret = -EFAULT;
	}
	if (alloc_ret == 0){
		unregister_chrdev_region(dev, enedev_devs);
        	ret = -EFAULT;
	}
#endif
 release3:
	kfree(keybd_io);

 release2:
	kfree(kb);

 release1:
	return ret;
}

static int __devexit kb3310_remove(struct i2c_client *client)
{
	struct kb3310_data *kb = i2c_get_clientdata(client);
#ifdef DEBUG_UDEV
	dev_t dev = MKDEV(enedev_major, 0);
#endif
	if (kb->enabled_ifaces & KB3310_USE_PS2_KEYBOARD)
		serio_unregister_port(kb->keybd_io);
	if (kb->enabled_ifaces & KB3310_USE_PS2_MOUSE)
		serio_unregister_port(kb->mouse_io);
	i2c_set_clientdata(client, NULL);
	kfree(kb);

#ifdef DEBUG_UDEV
	/* unregister class */
	device_destroy(enedev_class, enedev_dev);
	class_destroy(enedev_class);
	cdev_del(&enedev_cdev);
	unregister_chrdev_region(dev, enedev_devs);
	printk(KERN_ALERT "enedev driver removed.\n");	
#endif

	return 0;
}

#ifdef CONFIG_PM
/*
 * We don't need to explicitly suspend the chip, as it already switches off
 * when there's no activity.
 */
static int kb3310_suspend(struct i2c_client *client, pm_message_t mesg)
{
	struct kb3310_data *kb = i2c_get_clientdata(client);

	if ((kb->enabled_ifaces & KB3310_USE_PS2_KEYBOARD) ||
		(kb->enabled_ifaces & KB3310_USE_PS2_MOUSE)) {
		set_irq_wake(kb->client->irq, 0);
		disable_irq(kb->client->irq);
	}

	mutex_lock(&kb->lock);
	kb->pm_suspend = 1;
	mutex_unlock(&kb->lock);

	return 0;
}

static int kb3310_resume(struct i2c_client *client)
{
	struct kb3310_data *kb = i2c_get_clientdata(client);

	mutex_lock(&kb->lock);
	kb->pm_suspend = 0;
	mutex_unlock(&kb->lock);

	if ((kb->enabled_ifaces & KB3310_USE_PS2_KEYBOARD) ||
		(kb->enabled_ifaces & KB3310_USE_PS2_MOUSE)) {
		enable_irq(kb->client->irq);
		set_irq_wake(kb->client->irq, 1);
	}

	return 0;
}
#else
#define kb3310_suspend	NULL
#define kb3310_resume	NULL
#endif

static const struct i2c_device_id kb3310_id[] = {
	{ "kb3310", 0 },
	{ }
};

static struct i2c_driver kb3310_i2c_driver = {
	.driver = {
		.name	 = "kb3310",
		.owner  = THIS_MODULE,
 	},
	.probe		= kb3310_probe,
	.remove		= __devexit_p(kb3310_remove),
	.suspend	= kb3310_suspend,
	.resume		= kb3310_resume,
	.id_table	= kb3310_id,
};
MODULE_DEVICE_TABLE(i2c, kb3310_id);

static int __init kb3310_init(void)
{
	kb3310_wq = create_workqueue("kb3310");
	if (!kb3310_wq) {
		printk(KERN_ERR "Creation of kb3310_wq failed\n");
		return -EFAULT;
	}
	return i2c_add_driver(&kb3310_i2c_driver);
}

static void __exit kb3310_exit(void)
{
	i2c_del_driver(&kb3310_i2c_driver);
	destroy_workqueue(kb3310_wq);
}

module_init(kb3310_init);
module_exit(kb3310_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("XXX <xxx@marvell.com>");
MODULE_DESCRIPTION("ENE KB3310 PS/2 keyboard/mouse driver");
