#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# «locale-support»
#
# Ubiquity plugin for loading packages when a particular locale is picked. 
#
# Copyright (C) 2012, Canonical, Ltd.
#
# Author:
#  - Christopher Townsend <Christopher.Townsend@canonical.com>

import debconf
import os
import syslog
import subprocess
from ubiquity.plugin import *
from ubiquity import install_misc

NAME = 'locale-support'
AFTER = 'timezone'
WEIGHT = 12

CDROM_MOUNT = '/cdrom'

# Used as structure to define which packages should be installed
# or removed in one region
class RegionPackages:
    def __init__(self):
        self.region = ""
        self.to_install_packages = set()
        self.to_remove_packages = set()

# Global to save all the region packages.
region_packages_array = []

class Install(InstallPlugin):

    def get_client_type(self):
        client_type_file = ''
        # Assume the client type is consumer if the on image file is blank or
        # doesn't exist.
        client_type = 'consumer'

        if os.path.exists('/usr/lib/ubiquity/client_type'):
            # Here is a hook for downstream projects, in volatile-task
            # you could set it as 'kylin' or 'consumer' 
            client_type_file = '/usr/lib/ubiquity/client_type'
        elif os.path.exists('/cdrom/.oem/client_type'):
            client_type_file = '/cdrom/.oem/client_type'
        elif os.path.exists('/.oem/client_type'):
            client_type_file = '/.oem/client_type'
        else:
            return client_type

        with open(client_type_file, 'r') as rfd:
            client_type = rfd.readline()
            client_type = client_type.strip('\n')

        return client_type

    def load_region_packages_list(self, list_file):
        global region_packages_array

        region_packages = None

        with open(list_file, 'r') as lf:

            lines = lf.readlines()

            for line in lines:

                line = line.strip('\n')

                # '##' is keyword to start a new region
                if line.startswith("##?"):

                    if line.startswith("##? End:"):
                        # '## End:' is a keyword to exist
                        syslog.syslog("End all region!")
                        region_packages_array.append(region_packages)

                        # Close the file
                        lf.close()        
                        return

                    if region_packages != None:
                        # Finish one region packages, and begin a new region
                        syslog.syslog("Finish previous region packages, start a new: %s" % line.lstrip('##? '))
                        region_packages_array.append(region_packages)
                        region_packages = None

                    # Start a new region
                    syslog.syslog("Start a new region: %s" % line.lstrip('##? '))
                    # Create a new region when start a new region or end a old region
                    region_packages = RegionPackages()
                    region_packages.region=line.lstrip('##? ')

                elif line == '':
                    # Skip blank line
                    continue

                elif line.startswith("#"):
                    # Skip comment line
                    continue
                elif line.endswith("-"):
                    # To be removed packages in one region
                    syslog.syslog("To be removed packages in one region: %s" % line[:-1])
                    region_packages.to_remove_packages.add(line[:-1])
                else:
                    # To be installed packages in one region
                    syslog.syslog("To be installed packages in one region: %s" % line)
                    region_packages.to_install_packages.add(line)

    def eval_region(self, code, client_type, lang, encoding, timezone):
        # convert the return value to boolean
        return not not eval(code)

    def get_timezone():
        timezone_file = ''
        timezone = ''

        if os.path.exists('/etc/timezone'):
            timezone_file = '/etc/timezone'
        else:
            return timezone

        with open(timezone_file, 'r') as rfd:
            timezone = rfd.readline()
            timezone = timezone.strip('\n')

        return timezone

    def install(self, target, progress, *args, **kwargs):

        client_type = self.get_client_type()
        syslog.syslog("client_type is %s." % client_type)

        try:
            locale = progress.get('debian-installer/locale')
            # just use the prefix, such as use zh_CN for zh_CN.UTF-8
            syslog.syslog("locale is %s." % locale)
            if '.' in locale:
                lang, encoding = locale.split('.')
            else:
                lang = locale
                encoding = 'UTF-8'
        except debconf.DebconfError as e:
            syslog.syslog("WARNING: Failed to get locale!!!")
            lang = ''
            encoding = ''

        # TODO:
        # Currently it won't set into 'time/zone' when user select timezone in OOBE.
        # We need read '/etc/timezone', we need a method to judge country with timezone.
        try:
            timezone = progress.get('time/zone')
            syslog.syslog('timezone is %s' % timezone)
        except debconf.DebconfError as e:
            # Read the timezone from etc directory.
            timezone = self.get_timezone()

        # Now we use region_packages.list file to decide which packages should be installed or removed
        try:
            region_packages_file = progress.get('locale-support-plugin/region-packages-file')
        except debconf.DebconfError as e:
            region_packages_file = '/usr/lib/ubiquity/plugins/region_packages.list'

        # Downstream project could put the hooks to handle the region_packages.list or do something else
        subprocess.call("run-parts --report /usr/share/ubuntu/scripts/locale --arg=" + client_type + " --arg=" + lang + " --arg=" + encoding + " --arg=" + timezone + " >> /var/log/columbia-locale-plugin.log 2>&1", shell=True)

        self.load_region_packages_list(region_packages_file)

        for rule in region_packages_array:
            if self.eval_region(rule.region, client_type, lang, encoding, timezone) :
                if rule.to_install_packages :
                    syslog.syslog("To be installed packages %s." % str(rule.to_install_packages))
                    install_misc.record_installed(rule.to_install_packages)

                if rule.to_remove_packages :
                    syslog.syslog("To be removed packages %s." % str(rule.to_remove_packages))
                    install_misc.record_removed(rule.to_remove_packages)
            else:
                syslog.syslog("Skip this rule: %s." % rule.region)

