/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include "libgadget-host.h"

#include <glib.h>
#include <string>

#include <ggadget/decorated_view_host.h>
#include <ggadget/floating_main_view_decorator.h>
#include <ggadget/gadget.h>
#include <ggadget/gadget_consts.h>

#include <ggadget/gtk/main_loop.h>
#include <ggadget/gtk/utilities.h>
#include <ggadget/gtk/single_view_host.h>

#include <ggadget/extension_manager.h>
#include <ggadget/gadget_manager_interface.h>
#include <ggadget/host_utils.h>
#include <ggadget/main_loop_interface.h>
#include <ggadget/options_interface.h>
#include <ggadget/script_runtime_manager.h>
#include <ggadget/system_utils.h>

//static ggadget::gtk::MainLoop g_main_loop;
static const gchar socket_name[] = "ggl-host-socket";
static const char options_name[] = "gtk-host-options";

static const char *kGlobalExtensions[] = {
  "default-framework",
  "libxml2-xml-parser",
  "default-options",
  "dbus-script-class",
  "gtk-edit-element",
  "gtkmoz-browser-element",
  "gtk-flash-element",
  "gst-video-element",
  "gtk-system-framework",
  "gst-audio-framework",
  "linux-system-framework",
  "smjs-script-runtime",
  "curl-xml-http-request",
  "analytics-usage-collector",
  "google-gadget-manager",
  NULL
};

using namespace ggadget::gtk;

namespace ggadget {

void
LibgadgetHost::Init ()
{
  ExtensionManager     *ext_manager;
  ScriptRuntimeManager *script_runtime_manager;
  OptionsInterface     *options;
  std::string           profile_dir;
  std::string           error;
  static gboolean       inited = FALSE;

  if (inited)
    return;
  inited = TRUE;

  ggadget::SetGlobalMainLoop (new ggadget::gtk::MainLoop());

  /* Setup the gadget framework */
  profile_dir = ggadget::BuildFilePath (ggadget::GetHomeDirectory().c_str(),
                                        kDefaultProfileDirectory, NULL);
  ggadget::EnsureDirectories (profile_dir.c_str ());
  ggadget::SetupGlobalFileManager (profile_dir.c_str ());
  
  ggadget::SetupLogger (3, false);
  
  /* Setup extention manager */
  ext_manager = ExtensionManager::CreateExtensionManager ();
  ExtensionManager::SetGlobalExtensionManager (ext_manager);

  for (size_t i = 0; kGlobalExtensions[i]; ++i)
    ext_manager->LoadExtension (kGlobalExtensions[i], false);

  /* Load Javascript runtime */
  script_runtime_manager = ScriptRuntimeManager::get ();
  ScriptRuntimeExtensionRegister script_runtime_register(script_runtime_manager);
  ext_manager->RegisterLoadedExtensions (&script_runtime_register);

  if (!CheckRequiredExtensions (&error))
  {
    g_warning ("Unable to init the ggl framework: %s", error.c_str ());
  }

  /* Make the global extension manager readonly to avoid a bad gadget from 
   * registering local extensions into the global extension manager
   */
  ext_manager->SetReadonly ();

  InitXHRUserAgent ("libgadget-ggadgets");

  options = ggadget::CreateOptions (options_name);

  // ggadget::GetGadgetManager()->Init();
}

LibgadgetHost::LibgadgetHost (GGadget *gadget)
  :gadget_(gadget),
   lvh_(NULL)
{
  GadgetManagerInterface *manager = GetGadgetManager ();
  manager->Init ();
  
  global_permissions_.SetGranted (Permissions::ALL_ACCESS, true);
  global_permissions_.GrantAllRequired ();
}

LibgadgetHost::~LibgadgetHost ()
{
  ;
}

void
LibgadgetHost::OnMainViewCloseHandler() 
{
  g_signal_emit_by_name (gadget_, "remove-me");
}

ViewHostInterface *
LibgadgetHost::NewViewHost (Gadget                  *gadget,
                            ViewHostInterface::Type  type)
{
  if (type == ViewHostInterface::VIEW_HOST_MAIN)
  {
    FloatingMainViewDecorator *view_decorator;
    DecoratedViewHost *dvh;
    
    lvh_ = new LibgadgetViewHost (gadget_, type);
    view_decorator = new FloatingMainViewDecorator (lvh_, true);
    dvh = new DecoratedViewHost (view_decorator);
    view_decorator->SetButtonVisible(
            MainViewDecoratorBase::POP_IN_OUT_BUTTON, false);
    view_decorator->ConnectOnClose(
            NewSlot(this, &LibgadgetHost::OnMainViewCloseHandler));
    lvh_->ShowView (false, 0, NULL);

    return dvh;
  }
  else if (type == ViewHostInterface::VIEW_HOST_OPTIONS)
  {
    return new SingleViewHost (type, 1.0,
                           SingleViewHost::WM_MANAGEABLE
                           | SingleViewHost::DECORATED,
                           false);
  }
  else if (type == ViewHostInterface::VIEW_HOST_DETAILS)
  { 
   return new SingleViewHost (type, 1.0,
                           SingleViewHost::WM_MANAGEABLE
                           | SingleViewHost::DECORATED,
                           false);  
  }

  return NULL;
}

void
LibgadgetHost::RemoveGadget (Gadget *gadget,
                             bool    save_data)
{
  g_signal_emit_by_name (gadget_, "remove-me");
}

bool
LibgadgetHost::LoadFont (const char *filename)
{
  return false;
}

int
LibgadgetHost::GetDefaultFontSize ()
{
  return kDefaultFontSize;
}

bool
LibgadgetHost::OpenURL (const Gadget *gadget,
                        const char   *url)
{
  return ggadget::gtk::OpenURL (gadget, url);
}

void
LibgadgetHost::Draw (cairo_t *cr)
{
  if (lvh_)
    lvh_->Draw (cr);
}

Gadget *
LibgadgetHost::LoadGadget (const char *path,
                           const char *options_name_,
                           int         instance_id,
                           bool        show_debug)
{
  Gadget *gadget;
  StringMap manifest;
  if (!Gadget::GetGadgetManifest(path, &manifest)) {
    g_warning ("Unable to get Gadget manifest: %s", path);
    return NULL;
  }

  StringMap::const_iterator id_it = manifest.find(kManifestId);
  if (id_it == manifest.end() || id_it->second.empty()) {
    g_warning ("Gadget Manifest empty: %s", path);
    return NULL;
  }

  Permissions permissions;
  Gadget::GetGadgetRequiredPermissions(&manifest, &permissions);

  permissions.GrantAllRequired();

  //global_permissions_.SetGranted (Permissions::ALL_ACCESS, true);

  Gadget::SaveGadgetInitialPermissions (options_name_, permissions);

  gadget = new Gadget (this,
                       path,
                       options_name_,
                       0,
                       global_permissions_,
                       Gadget::DEBUG_CONSOLE_DISABLED);

  if (!gadget->IsValid ())
  {
    g_warning ("Failed to load gadget %s", path);
    delete gadget;
    return NULL;
  }

  gadget->SetDisplayTarget(Gadget::TARGET_FLOATING_VIEW);
  gadget->GetMainView()->OnOtherEvent(SimpleEvent(Event::EVENT_UNDOCK));

  if (!gadget->ShowMainView ())
  {
    g_warning ("Failed to show main view of gadget %s", path);
    delete gadget;
    return NULL;
  }  
  return gadget;
}

} /* namespace ggadget */
