/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * Authors:
 *  Olivier Tilloy <olivier.tilloy@canonical.com>
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; version 3.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "bamf-matcher.h"
#include "bamf-matcher-proxy.h"
#include "bamf-factory.h"
#include "bamf-view.h"
#include "bamf-application.h"
#include "bamf-window.h"

BamfMatcher::BamfMatcher() :
    m_proxy(NULL)
{
    m_proxy = new OrgAyatanaBamfMatcherInterface("org.ayatana.bamf",
                                                 "/org/ayatana/bamf/matcher",
                                                 QDBusConnection::sessionBus(),
                                                 static_cast<QObject*>(this));
    // Forward signals from the proxy
    connect(m_proxy, SIGNAL(ActiveApplicationChanged(const QString &, const QString &)),
            this, SLOT(OnActiveApplicationChanged(const QString &, const QString &)));
    connect(m_proxy, SIGNAL(ActiveWindowChanged(const QString &, const QString &)),
            this, SLOT(OnActiveWindowChanged(const QString &, const QString &)));
    connect(m_proxy, SIGNAL(ViewClosed(const QString &, const QString &)),
            this, SLOT(OnViewClosed(const QString &, const QString &)));
    connect(m_proxy, SIGNAL(ViewOpened(const QString &, const QString &)),
            this, SLOT(OnViewOpened(const QString &, const QString &)));
}

BamfMatcher::~BamfMatcher()
{
    delete m_proxy;
}

BamfApplication *
BamfMatcher::active_application() const
{
    QDBusPendingReply<QString> reply = m_proxy->ActiveApplication();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return NULL;
    } else {
        QString path = reply.value();
        if (path.isEmpty())
            return NULL;

        BamfView *view = BamfFactory::get_default().view_for_path(path);
        return qobject_cast<BamfApplication *>(view);
    }
}

BamfWindow *
BamfMatcher::active_window() const
{
    QDBusPendingReply<QString> reply = m_proxy->ActiveWindow();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return NULL;
    } else {
        QString path = reply.value();
        if (path.isEmpty())
            return NULL;

        BamfView *view = BamfFactory::get_default().view_for_path(path);
        return qobject_cast<BamfWindow *>(view);
    }
}

BamfApplication *
BamfMatcher::application_for_xid(uint xid) const
{
    QDBusPendingReply<QString> reply = m_proxy->ApplicationForXid(xid);
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return NULL;
    } else {
        QString path = reply.value();
        if (path.isEmpty())
            return NULL;

        BamfView *view = BamfFactory::get_default().view_for_path(path);
        return qobject_cast<BamfApplication *>(view);
    }
}

bool
BamfMatcher::application_is_running(const QString& application) const
{
    QDBusPendingReply<bool> reply = m_proxy->ApplicationIsRunning(application);
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
        return false;
    } else {
        return reply.value();
    }
}

BamfApplicationList *
BamfMatcher::applications() const
{
    QList<BamfApplication *> result;
    QDBusPendingReply<QStringList> reply = m_proxy->ApplicationPaths();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
    } else {
        QStringList list = reply.value();
        for (int i = 0; i < list.size(); ++i) {
            BamfView *view = BamfFactory::get_default().view_for_path(list.at(i));
            BamfApplication* application = qobject_cast<BamfApplication *>(view);
            if (application != NULL) {
                result.prepend(application);
            }
        }
    }
    return new BamfApplicationList(result);
}

BamfApplicationList *
BamfMatcher::running_applications() const
{
    QList<BamfApplication *> result;
    QDBusPendingReply<QStringList> reply = m_proxy->RunningApplications();
    reply.waitForFinished();
    if (reply.isError()) {
        qWarning() << reply.error();
    } else {
        QStringList list = reply.value();
        for (int i = 0; i < list.size(); ++i) {
            BamfView *view = BamfFactory::get_default().view_for_path(list.at(i));
            BamfApplication* application = qobject_cast<BamfApplication *>(view);
            if (application != NULL) {
                result.prepend(application);
            }
        }
    }
    return new BamfApplicationList(result);
}

void
BamfMatcher::register_favorites(const QStringList& favorites) const
{
    QDBusPendingReply<> reply = m_proxy->RegisterFavorites(favorites);
    reply.waitForFinished();

    if (reply.isError()) {
        qWarning() << reply.error();
    }
}

void
BamfMatcher::OnActiveApplicationChanged(const QString &old_active, const QString &new_active)
{
    BamfView *former = BamfFactory::get_default().view_for_path(old_active);
    BamfView *current = BamfFactory::get_default().view_for_path(new_active);
    ActiveApplicationChanged(qobject_cast<BamfApplication *>(former),
                             qobject_cast<BamfApplication *>(current));
}

void
BamfMatcher::OnActiveWindowChanged(const QString &old_active, const QString &new_active)
{
    BamfView *former = BamfFactory::get_default().view_for_path(old_active);
    BamfView *current = BamfFactory::get_default().view_for_path(new_active);
    ActiveWindowChanged(qobject_cast<BamfWindow *>(former),
                        qobject_cast<BamfWindow *>(current));
}

void
BamfMatcher::OnViewClosed(const QString &path, const QString &type)
{
    Q_UNUSED(type)
    BamfView *view = BamfFactory::get_default().view_for_path(path);
    ViewClosed(view);
}

void
BamfMatcher::OnViewOpened(const QString &path, const QString &type)
{
    Q_UNUSED(type)
    BamfView *view = BamfFactory::get_default().view_for_path(path);
    ViewOpened(view);
}

