/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	Copyright 1998-2002 Anton Vinokurov <anton@netams.com>
***	Copyright 2002-2008 NeTAMS Development Team
***	This code is GPL v3
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: flowengine.c,v 1.5 2008-05-27 17:14:15 jura Exp $ */

#include "netams.h"
#include "ds_any.h"

#define FLOW_HASH(idx)  CRC16(idx, FLOW_IDX_SIZE);
//////////////////////////////////////////////////////////////////////////
u_char DSsystem(NetUnit *u, dsUlist *start);
//////////////////////////////////////////////////////////////////////////
FlowEngine::FlowEngine(Service_DS *ds) {
	//FlowEngine should know about friends
	instance=ds->instance;
	max_flow_slots = ds->max_flow_slots;

	IPtree=ds->IPtree;
	fifo=ds->ds_fifo;
	
	flags = 0;

	e_used=e_total=0;

	t=t_now=time(NULL);
	
	last_id = expired_id = sent_flows = 0;

#ifdef HAVE_BW
	BW=NULL;
#endif
	ready=NULL;
	active=NULL;
	table=(entry_hash*)aMalloc(FLOW_HASH_SIZE*sizeof(entry_hash));
	aLog(D_INFO, "FLOW HASH initialized with size %u\n",FLOW_HASH_SIZE);

	message=(EngineMsg*)aMalloc(sizeof(EngineMsg));
	aLog(D_INFO, "Flow engine started\n");
}
//////////////////////////////////////////////////////////////////////////
FlowEngine::~FlowEngine() {
	FlushAll();
	entry *ptr=ready;
	while(ready) {
		ptr=ready;
		ready=ready->next;
		delete ptr->flow;
		aFree(ptr);
	}
#ifdef HAVE_BW	
	if(BW) delete BW;
#endif
	if(message->flow) delete message->flow;
	aFree(message);
	aFree(table);
	aLog(D_INFO, "Flow Engine uninitialized\n");
}
//////////////////////////////////////////////////////////////////////////
void FlowEngine::CheckExpire(entry *e){
    	if (unsigned(e->flow_info->flow_last - e->flow_info->flow_first) > ACTIVE_TIMEOUT || 
	unsigned(t_now - e->flow_info->flow_last) > INACTIVE_TIMEOUT)  e->flags|=ENTRY_EXPIRED;
#ifdef DEBUG 
   	if(e->flags&ENTRY_EXPIRED) aDebug(DEBUG_FLOW,"%p expired\n", e);
#endif
}
//////////////////////////////////////////////////////////////////////////
int FlowEngine::Process(u_char *flow_key, u_long packets, u_long octets, entry **e2get) {
    	unsigned hash=FLOW_HASH(flow_key);
    	
	entry *e;
	entry_hash *h = &table[hash];
	u_short chunk=0;
    	
	for(e=h->root; e!=NULL; e=e->next){
        	if (!memcmp(e->index, flow_key, FLOW_IDX_SIZE)) {
            		e->flow_info->octets	+= octets;
                	e->flow_info->packets	+= packets;

#ifdef FAST_FW_CHECK
			register time_t last    = e->flow_info->flow_last;
			e->flow_info->flow_last = t_now;

                	if((flags&FLOW_FW_CHECK) && IS_FW_CHECK_CHANGED(last))
				FW(e);
#else
			e->flow_info->flow_last = t_now;
#endif


			if((e->flags&ENTRY_BLOCKED)
#ifdef HAVE_BW
			|| BWCheck(e, octets, tv)
#endif
			)
				return 0; //drop packet
			break;
        	}
		chunk++;
    	}
	
	if (e==NULL) {
		if(e_used > (MAX_UNITS<<7)) {
			aLog(D_WARN, "MAX entries limit reached (%u), fflushing active flows\n", MAX_UNITS<<7);
			FlushAll(); //protection against DoS
		} else if(chunk>MAX_CHUNK) {
			aLog(D_WARN, "MAX chunks limit reached (%u), fflushing active flows\n", MAX_CHUNK);
			FlushAll(); //protection against DoS
		}

                if(ready) { //we'll use ready empty entry
                        e=ready;
                        ready=ready->next;
                } else { // we must create a new flow record
                        e=(entry*)aMalloc(sizeof(entry));
			e->flow = new Flow(instance, max_flow_slots);
			
			e_total++;
                } 
			
		bcopy(flow_key, e->index, FLOW_IDX_SIZE);
		e->flow->reuse();

		e_used++;
		e->flow_info			= (struct flow_info_value*)e->flow->put(ATTR_FLOW_INFO);
        	e->flow_info->flow_first	= e->flow_info->flow_last=t_now;
		e->flow_info->packets		= packets;
		e->flow_info->octets		= octets;
#ifdef LAYER7_FILTER		
		e->layer7_info			= NULL;
#endif
        	e->flags=0;
		last_id++;
        
		if(h->root == NULL) {
			//organize active list
			h->next_active = active;
			active = h;
		}

		//plug new entry
		e->next=h->root;
		h->root=e;
		
#ifdef DEBUG
                aDebug(DEBUG_FLOW,"new entry %u [%u]-%p\n", last_id, hash, e);
#endif

    		*e2get=e;
		return -1; //no such entry -> give entry to fill
	}

	*e2get=e;
	return 1; //pass packet
}
//////////////////////////////////////////////////////////////////////////
void FlowEngine::Expiresearch(struct timeval *ttv){
	tv=ttv;

	t_now=tv->tv_sec;
	if(unsigned(t_now-t)<CHECK_EXPIRE) return;
	t=t_now;
	
        entry_hash *prev_h=NULL;
	entry_hash *next_h;

    	aDebug(DEBUG_FLOW,"ExpireSearch t:[%u] a:[%u] at %lu: \n", last_id,last_id-expired_id, t_now);
	
	for(entry_hash *h = active; h!=NULL; h = next_h) {
	    next_h = h->next_active;
	    
	    entry *prev_e=NULL;
	    entry *next_e;

	    for(entry *e = h->root; e!=NULL; e = next_e) {
	    	next_e = e->next;
		
		CheckExpire(e);
            	if (e->flags&ENTRY_EXPIRED) {
			// with high probability used packet might be used again thou ...
			if(e->flow_info->packets) {
				if(!(e->flags&ENTRY_BLOCKED)) DoSend(e);
#ifndef FAST_FW_CHECK
				//we need to recheck blocked status   
				if((flags&FLOW_FW_CHECK) && IS_FW_CHECK_CHANGED(e->flow_info->flow_first))
					FW(e);
#endif
				e->flow_info->packets=0;
				e->flow_info->octets=0;	
				e->flags&= ~ENTRY_EXPIRED;
				e->flow_info->flow_first=e->flow_info->flow_last=t_now;			
				aDebug(DEBUG_FLOW, "entry %p reused after expiration\n", e);
			} else {
				expired_id++;
				
				//remove from hash
				if(e == h->root) h->root = e->next;
				else prev_e->next = e->next;

				//this means we have flow that was unaccessible one round
				//move it to ready 
				e->next=ready;
				ready=e;
				e_used--;
#ifdef HAVE_BW
				BW->FreeBWentry(e);
#endif			
				aDebug(DEBUG_FLOW, "entry %p moved to ready\n", e);
				continue;
			}
            	}
		prev_e = e;
	    }
	    //remove non acive brances from list
	    if(h->root == NULL) {
	    	if(h == active) active = h->next_active;
		else prev_h->next_active = h->next_active;
	    } else 
	    	prev_h = h;
        }
}
//////////////////////////////////////////////////////////////////////////
void FlowEngine::FlushAll() {
	entry *next_e;

	aDebug(DEBUG_FLOW,"FlushAll::begin\n");
	aDebug(DEBUG_FLOW,"FlushAll t:[%u] a:[%u] at %lu: \n", last_id, last_id-expired_id, t_now);
	
	for (entry_hash *h = active; h!=NULL; h = h->next_active) {
		for(entry *e = h->root; e!=NULL; e = next_e) {
			next_e = e->next;

			expired_id++;
			
			if(e->flow_info->packets && !(e->flags&ENTRY_BLOCKED)) DoSend(e);
			e->next=ready;
			ready=e;
			e_used--;
#ifdef HAVE_BW
			BW->FreeBWentry(e); //flush BW slots
#endif 
		}
		h->root=NULL;
	}
	active=NULL;
	aDebug(DEBUG_FLOW,"FlushAll::end\n");
}
//////////////////////////////////////////////////////////////////////////
void FlowEngine::DoSend(entry *e) {
	DoSend(e->flow);

#ifdef LAYER7_FILTER
	if(e->layer7_info) {
		e->layer7_info->value=NULL;
		e->layer7_info=NULL;
	}
#endif
}

void FlowEngine::DoSend(Flow *flow) {
#ifdef DEBUG	
	if(aDebug(DEBUG_FLOW,"sending flow %p \n", flow)) {
		flow->Debug(DEBUG_FLOW);
	}
#endif
	
	if(flow->freeSlot > max_flow_slots) {
		aLog(D_WARN, "Maximum flow slots increased from %u to %u\n", max_flow_slots, flow->freeSlot);
		max_flow_slots = flow->freeSlot;
	}

	//transfer to s_datasource here
	if(!message->flow)
		message->flow = new Flow(instance, max_flow_slots);

	message->flow->copy(flow);
	message=fifo->Push(message);
	sent_flows++;
}
//////////////////////////////////////////////////////////////////////////
void FlowEngine::InitFW(u_char fw_check) {

	if(fw_check) {
		flags|=FLOW_FW_CHECK;
#ifdef HAVE_BW
		if(!BW) BW = new BWEngine();
#endif	
	} else {
		flags&=~FLOW_FW_CHECK;
#ifdef HAVE_BW		
		if(BW) delete BW;
		BW=NULL;
#endif
	}
}
//////////////////////////////////////////////////////////////////////////
#define FW_UNSET	0x00
#define FW_PASS		0x01
#define FW_DROP		0x02
#define FW_BACKUP	0x04
#define FW_BRK		0x10
//////////////////////////////////////////////////////////////////////////
u_char FlowEngine::FW(entry *e) {
	NetUnit *u;
    	dsUlist *dsu;
	
	current_entry=e;
	Flow *flow=e->flow;
    	u_char ufw_res, fw_res=FW_UNSET;    //Processor->restrict_all; // 0=pass, 1=drop

	DISABLE_CANCEL;
    	netams_rwlock_rdlock(&IPtree->rwlock);

    	//here we organize units that matches packet
    	dsu=current_dsulist=IPtree->checktree(flow, CHECK_FW);

	for(;dsu!=NULL;dsu=dsu->link[CHECK_FW]) {
    		if(fw_res&FW_BRK) { //we know we drop packet, but we need to clear mf flag
        		dsu->mf[CHECK_FW]=MATCH_NONE;
	    		continue;
        	}
        
		current_mf=dsu->mf[CHECK_FW];
		ELIST_FOR_EACH(dsu->list, u) {
			ufw_res=FW_PASS;
			
			//check packet might be processed incorrectly due to wrong match flag
            		if(u->flags&NETUNIT_BROKEN_FW_MF) {
            			match mmf=u->Check(flow);
                		if(mmf!=current_mf && current_mf==MATCH_DST)
                			continue;
                
				current_mf=mmf;
            		} else
                		current_mf=dsu->mf[CHECK_FW];

            		// yes, this unit corresponds to this data packet
            		aDebug(DEBUG_FLOW, "packet matches unit %s (oid=%06X)\n", u->name, u->id);
			if(DSfw(u) == DROP)
				ufw_res=FW_BRK|FW_DROP;

			if(u->flags&NETUNIT_BACKUP_FW) { //backup-fw
				aDebug(DEBUG_FLOW, "This unit has backup-fw, use fw decision only as last resort\n");
				if(fw_res==FW_UNSET ||
			 	(fw_res==(FW_BACKUP|FW_PASS))) {
					//check decision already made or previous backup
					//if we have one backup fw already
					//and it's PASS we should overwrite it
					ufw_res&=~FW_BRK;	//strip BRK flag
					ufw_res|=FW_BACKUP;     //mark this decision as backup
				} else	
					continue; 	//decision already made
			}

			if((u->flags&NETUNIT_NLP) && (ufw_res&FW_PASS)) { //no-local-pass
				aDebug(DEBUG_FLOW, "This unit has no-local-pass flag set, ignore PASS\n");
			} else 
				fw_res=ufw_res;
        	}
        	dsu->mf[CHECK_FW]=MATCH_NONE;
	}
    	netams_rwlock_unlock(&IPtree->rwlock);
    	RESTORE_CANCEL;

	if(fw_res==FW_UNSET)
		fw_res=Processor->restrict_all?FW_DROP:FW_PASS;	//Processor->restrict_all: 0=pass, 1=drop

	//make decision
	if(fw_res&FW_DROP) {
		e->flags|=ENTRY_BLOCKED;
		aDebug(DEBUG_FLOW, "Forward decision DROP\n");
		return 0;
	} else {
		e->flags&=~ENTRY_BLOCKED;
		aDebug(DEBUG_FLOW, "Forward decision PASS\n");
		return 1;
	}
}

u_char FlowEngine::DSfw(NetUnit *u){
	// we will perform the fw check for this unit
	if(DSsystem(u, current_dsulist)) {
		aDebug(DEBUG_FLOW, " fw check for %s %s: SYS-DROP\n", netunit_type_name[u->type], u->name);
		return DROP;
	}

   	if(!u->fp)
		return Processor->restrict_local;
	
	Flow *flow      = current_entry->flow;
	u_char fw_res	= DROP;

    for (policy_data *cpd=u->fp->root; cpd!=NULL; cpd=cpd->next) {
    	cpd->check++;
        if(cpd->policy->Check(flow, current_mf)) {
        	if (!(cpd->policy_flags&POLICY_FLAG_INV)) {
				cpd->match++;
				fw_res=PASS;
#ifdef HAVE_BW
				if(cpd->policy->bwi) {
					BW->AddBW2entry(cpd->policy->bwi, current_entry, current_mf, cpd);
				}
				if (cpd->policy_flags&POLICY_FLAG_BRK)
#endif
					break;
            }
		} else {
        	if (cpd->policy_flags&POLICY_FLAG_INV) {
				cpd->match++;
				fw_res=PASS;
#ifdef HAVE_BW
				if(cpd->policy->bwi) {
					BW->AddBW2entry(cpd->policy->bwi, current_entry, current_mf, cpd);
				}
				if (cpd->policy_flags&POLICY_FLAG_BRK)
#endif
					break;
			}
		}
	}

       	if(fw_res == DROP) {
       		aDebug(DEBUG_FLOW, " fw check for %s %s: DROP\n", netunit_type_name[u->type], u->name);
		return DROP;
	}

	// now check the fw policy for all parents (groups)
	NetUnit_group *up;
	ELIST_FOR_EACH(u->parents, up)
		if (up->checkDSList(instance) && DSfw(up)==DROP)
			return DROP;

#ifdef HAVE_BW
	//add bw for this unit
	if(u->bwi)
		BW->AddBW2entry(u->bwi, current_entry, current_mf, u->bwi);
#endif
	return PASS;
}

u_char DSsystem(NetUnit *u, dsUlist *dsu) {
        if(!u->sys_policy_perm) 
		return u->sys_policy;

	NetUnit *d;
        for(;dsu!=NULL;dsu=dsu->link[CHECK_FW]) {
		ELIST_FOR_EACH(dsu->list, d) {
                	if (d!=u && u->sys_policy_perm==d->id)
                        	return (u->sys_policy&SP_DENY);
		}
	}
        return (u->sys_policy&~SP_DENY);
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
struct ipv4_key* IPv4GetKey(struct ip *ip, struct ipv4_key* ipv4_flow_key){ 

	ipv4_flow_key->ipv4_info.ip_p    = ip->ip_p;
	ipv4_flow_key->ipv4_info.ip_tos  = ip->ip_tos;
	ipv4_flow_key->ipv4_info.ip_src  = ip->ip_src;
	ipv4_flow_key->ipv4_info.ip_dst  = ip->ip_dst;

	if(ip->ip_p == IPPROTO_TCP || ip->ip_p == IPPROTO_UDP) {
		struct tcphdr       *th = (struct tcphdr *)((u_char *)ip + ip->ip_hl*4);
		
		ipv4_flow_key->tcp_info.src_port	= th->th_sport;
		ipv4_flow_key->tcp_info.dst_port	= th->th_dport;
	} else {
		ipv4_flow_key->tcp_info.src_port	= 0;
		ipv4_flow_key->tcp_info.dst_port	= 0;
	}
	return ipv4_flow_key;
}
void IPv4FillFlow(struct ipv4_key* ipv4_flow_key, entry *flow_entry) {
	flow_entry->flow->put(ATTR_IPV4_INFO, (union attribute_value*)&ipv4_flow_key->ipv4_info);
	
	if(ipv4_flow_key->ipv4_info.ip_p == IPPROTO_TCP || ipv4_flow_key->ipv4_info.ip_p == IPPROTO_UDP) {
		flow_entry->flow->put(ATTR_TCP_INFO, (union attribute_value*)&ipv4_flow_key->tcp_info);
	}
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
// DS_FIFO class

DS_FIFO::DS_FIFO(unsigned max){
        root=last=ready=NULL;
        num_items=total_items=ready_items=0;
        max_items=max;
        netams_mutex_init(&lock, NULL);
}

DS_FIFO::~DS_FIFO(){
	EngineMsg *m,*ptr=ready;
        ready=NULL;
        while(ptr) {
                m=ptr->next;
		if(ptr->flow) delete ptr->flow;
                aFree(ptr);
                ptr=m;
        }

        netams_mutex_destroy(&lock);
}

EngineMsg* DS_FIFO::Push(EngineMsg *msg){
        netams_mutex_lock(&lock);
        if (root==NULL) root=msg;
        else last->next=msg;
        last=msg;
        msg->next=NULL;
        num_items++;
        total_items++;
        if (num_items>max_items) {
		max_items+=1000;
                aLog(D_WARN, "FIFO %p overloaded, increasing max_items to %u\n",this,max_items);
/*                msg=root;
                root=root->next;
                aFree(msg);
                num_items--;
*/
        }
	// prepare message
	if(ready) {
		msg=ready;
		ready=ready->next;
		ready_items--;
	} else {
		msg=(EngineMsg*)aMalloc(sizeof(EngineMsg));
	}
        netams_mutex_unlock(&lock);
//      printf("FIFO push: %p\n", msg);
        return msg;
}

EngineMsg* DS_FIFO::Pop(EngineMsg *msg){
        netams_mutex_lock(&lock);
	// put used message in ready
	if(msg) {
		msg->next=ready;
		ready=msg;
		ready_items++;
	}

        msg=root;
        if (root) {
		root=root->next;
        	num_items--;
	}
        netams_mutex_unlock(&lock);
//      printf("FIFO pop:  %p\n", m);
        return msg;
}
//////////////////////////////////////////////////////////////////////////
