/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n.h>

#include <gio/gio.h>
#include <glib/gstdio.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include <libwnck/libwnck.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include <libgnome/libgnome.h>
#include <libgnomeui/libgnomeui.h>
#include <libgnome/gnome-desktop-item.h>

#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>
#include <clutter/x11/clutter-x11.h>

#include <libnotify/notify.h>

#include "nl-app.h"
#include "nl-app-loader.h"
#include "nl-favorite-item.h"
#include "nl-folders-source.h"
#include "nl-menu-items.h"
#include "nl-plugin-manager.h"
#include "nl-uri-loader.h"
#include "nl-volumes-source.h"
#include "nl-window.h"

#define SYSTEM_FAV_MENU PKGDATADIR"/Favorites.menu"
#define LOCAL_FAV_DIR  ".config/menus/applications-merged"
#define LOCAL_FAV_MENU ".config/menus/applications-merged/Favorites.menu"

#define APP_NAME "Netbook Launcher"

/* Forwards */
//static void     on_system_resume                 (NlConfig *config);

static void     check_required_files_and_folders (void);
static gboolean check_old_config_before_running  (void);
static void     default_signal_handler           (gint sig);

/* I know, globals */
gboolean      windowed      = FALSE;
gint          window_width  = 1024;
gint          window_height = 576;
gboolean      norestart     = FALSE;
static gchar *add_favorite = NULL;

static gboolean check_run = FALSE;

static GOptionEntry entries[] =
{
  {
    "windowed",
    'w', 0,
    G_OPTION_ARG_NONE,
    &windowed,
    "Launch in windowed mode (for testing - 1024x600)",
    NULL
  },
  {
    "width",
    'd', 0,
    G_OPTION_ARG_INT,
    &window_width,
    "width",
    NULL
  },
  {
    "height",
    'h', 0,
    G_OPTION_ARG_INT,
    &window_height,
    "height",
    NULL
  },
  {
    "add-favorite",
    'a', 0,
    G_OPTION_ARG_STRING,
    &add_favorite,
    "Path of favorite to add"
  },
  {
    "no-restart",
    'r', 0,
    G_OPTION_ARG_NONE,
    &norestart,
    "Do not restart on VT changes",
    NULL
  },
  {
    "check-run",
    'c', 0,
    G_OPTION_ARG_NONE,
    &check_run,
    "This option is available to facilitate the upgrade from ume-nl to "
    "netbook-launcher. If enabled, it will check the existance of the "
    "ume-nl.desktop autostart suppression file in ~/.config/autostart "
    "prior to starting, and replace it with one for netbook-launcher, so "
    "launchers that upgraded while in Classic mode do not have a nl popup.",
    NULL
  },
  {
    NULL
  }
};

static void
on_session_quit (GnomeClient *client)
{
  g_debug ("Quitting nicely");
  gtk_widget_destroy (nl_window_get_default ());
  gtk_main_quit ();
}

static gboolean
on_session_save (GnomeClient *client)
{
  g_debug ("Saved");
  return TRUE;
}

gint
main (gint argc, gchar *argv[])
{
  GOptionContext        *context;
  GError                *error = NULL;
  NlApp                 *app;
  GtkWidget             *window;
  GObject               *folders_source;
  GObject               *volumes_source;
  GObject               *app_loader;
  GObject               *uri_loader;
  NlMenuItems           *menu_items;
  NlPluginManager       *plugin_manager;
  NlFavoriteItem        *favorite_item;
  GnomeClient           *client;

  NlConfig *cfg = NULL;

  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  if (!notify_init ("Netbook Launcher"))
    return EXIT_FAILURE;

  if (!g_thread_supported ())
    g_thread_init (NULL);
  g_set_application_name (APP_NAME);
  gtk_clutter_init (&argc, &argv);

  context = g_option_context_new ("- " APP_NAME);
  g_option_context_add_main_entries (context, entries, APP_NAME);
  g_option_context_parse (context, &argc, &argv, &error);
  g_option_context_free (context);

  if (error)
    {
      g_print ("Unable to run " APP_NAME ": %s", error->message);
      g_error_free (error);
      return EXIT_FAILURE;
    }

  /* Check the existance of the ume-nl.desktop suppression file
   * before starting */
  if (check_run)
    {
      if (check_old_config_before_running ())
        return EXIT_SUCCESS;
    }
  check_required_files_and_folders ();

  gnome_program_init ("netbook-launcher", "0.1.2", LIBGNOMEUI_MODULE,
                      argc, argv, GNOME_PARAM_NONE, NULL);
  client = gnome_master_client ();
  g_signal_connect (client, "die",
                    G_CALLBACK (on_session_quit), NULL);
  g_signal_connect (client, "save-yourself",
                    G_CALLBACK (on_session_save), NULL);

  /* DO NOT REMOVE, that is, unless
   * https://bugs.freedesktop.org/show_bug.cgi?id=17327 is fixed properly on
   * Intel systems
   */
  clutter_set_font_flags (0);

  /* If we're in low graphics mode, disable motion events */
  if (windowed)
    nl_config_set_windowed (window_width, window_height);

  cfg = nl_config_get_default ();
  if (cfg->low_graphics)
    clutter_set_motion_events_enabled (FALSE);

  /* Make sure we're the only netbook-launcher instance running */
  app = nl_app_get_default ();
  if (unique_app_is_running (UNIQUE_APP (app)))
    {
      UniqueResponse     response;
      UniqueMessageData *data;

      /* Send the right message */
      if (add_favorite)
        {
          data = unique_message_data_new ();
          unique_message_data_set_text (data, add_favorite, -1);
          response = unique_app_send_message (UNIQUE_APP (app),
                                              COMMAND_ADD_FAVORITE,
                                              data);
          unique_message_data_free (data);
          g_free (add_favorite);
        }

      g_object_unref (app);
      g_object_unref (cfg);
      notify_uninit ();
      return EXIT_SUCCESS;
    }
  else
    nl_app_init_real (app);
  /* Create the nl window */
  window = nl_window_get_default ();

  plugin_manager = nl_plugin_manager_new (NL_SHELL (window));
  folders_source = nl_folders_source_new (NL_SHELL (window));
  volumes_source = nl_volumes_source_new (NL_SHELL (window));
  menu_items     = nl_menu_items_new (NL_SHELL (window));
  app_loader     = nl_app_loader_new (NL_SHELL (window));
  favorite_item  = nl_favorite_item_new (NL_SHELL (window));

  /* Make sure this is last, as it's kind of a 'catch-all' loader */
  uri_loader     = nl_uri_loader_new (NL_SHELL (window));

  gtk_widget_show (window);

  signal (SIGHUP, default_signal_handler);
  signal (SIGINT, default_signal_handler);
  signal (SIGTERM, default_signal_handler);
  signal (SIGUSR1, default_signal_handler);
  signal (SIGUSR2, default_signal_handler);

  //g_timeout_add (3000, (GSourceFunc)on_session_quit, NULL);

  gtk_main ();
  
  if (GTK_IS_WIDGET (window))
    gtk_widget_destroy (GTK_WIDGET (window));

  g_object_unref (app_loader);
  g_object_unref (favorite_item);
  g_object_unref (menu_items);
  g_object_unref (folders_source);
  g_object_unref (volumes_source);
  g_object_unref (uri_loader);

  g_object_unref (plugin_manager);

  g_object_unref (app);

  g_object_unref (cfg);

  notify_uninit ();

  return EXIT_SUCCESS;
}

static void
default_signal_handler (gint sig)
{
  g_debug ("Signal caught: %d", sig);
  on_session_quit (NULL);
}

/*
 * Creates any required folders and files for the nl
 */
static void
check_required_files_and_folders (void)
{
  gchar *local_apps;

  local_apps = g_build_filename (g_get_home_dir (),
                                 ".local", "share", "applications", NULL);
  if (!g_file_test (local_apps, G_FILE_TEST_EXISTS))
    {
      /* As the local application dir doesn't exist, we assume this is first
       * run
       */
      g_mkdir_with_parents (local_apps, 0700);
      g_debug ("Created local applications directory at %s", local_apps);
    }
  g_free (local_apps);

  local_apps = g_build_filename (g_get_user_config_dir (),
                                 "netbook-launcher", "cache", NULL);
  if (!g_file_test (local_apps, G_FILE_TEST_EXISTS))
    {
      /* As the local application dir doesn't exist, we assume this is first
       * run
       */
      g_mkdir_with_parents (local_apps, 0700);
      g_debug ("Created local applications directory at %s", local_apps);
    }
  g_free (local_apps);

  local_apps = g_build_filename (g_get_home_dir (),
                                 ".config", "netbook-launcher", "icons", NULL);
  if (!g_file_test (local_apps, G_FILE_TEST_EXISTS))
    {
      g_mkdir_with_parents (local_apps, 0700);
      g_debug ("Created local icons directory at %s", local_apps);
    }
  g_free (local_apps);

  local_apps = g_build_filename (g_get_home_dir (), LOCAL_FAV_MENU, NULL);
  if (!g_file_test (local_apps, G_FILE_TEST_EXISTS))
    {
      GFile *from, *to;
      gchar *temp = g_build_filename (g_get_home_dir (), LOCAL_FAV_DIR, NULL);
      g_mkdir_with_parents (temp, 0700);

      from = g_file_new_for_path (SYSTEM_FAV_MENU);
      to = g_file_new_for_path (local_apps);

      g_file_copy (from, to, 0, NULL, NULL, NULL, NULL);

      g_object_unref (from);
      g_object_unref (to);
    }
  g_free (local_apps);
}

/*
 * Check if there is a ~/.config/autostart/ume-nl.desktop autostart
 * suppression file present before starting (as we may be in classic mode).
 *
 */
static gboolean
check_old_config_before_running ()
{
#define AUTOSTART_ITEM "X-GNOME-Autostart-enabled"
  gchar *ume_desktop;

  ume_desktop = g_build_filename (g_get_home_dir (),
                                  ".config",
                                  "autostart",
                                  "ume-nl.desktop",
                                  NULL);

  if (g_file_test (ume_desktop, G_FILE_TEST_EXISTS))
    {
      GError           *error = NULL;
      GnomeDesktopItem *item = NULL;
      gchar            *netbook_desktop;
      gboolean          autostart_enabled = TRUE;

      item = gnome_desktop_item_new_from_file (ume_desktop,
             GNOME_DESKTOP_ITEM_LOAD_ONLY_IF_EXISTS,
             &error);


      if (error || !item)
        {
          g_warning ("Unable to read ume-nl.desktop at %s: %s",
                     ume_desktop, error ? error->message: " ");

          g_error_free (error);
          g_free (ume_desktop);
          return FALSE;
        }

      /* Check if the autostart file was to autostart or to suppress autostart */
      if (gnome_desktop_item_attr_exists (item, AUTOSTART_ITEM))
        {
          autostart_enabled = gnome_desktop_item_get_boolean (item, AUTOSTART_ITEM);
        }

      /* Update the exec string */
      gnome_desktop_item_set_string (item,
                                     GNOME_DESKTOP_ITEM_EXEC,
                                     "netbook-launcher");
      /* Save in new location */
      netbook_desktop = g_strdup_printf (
                          "file://%s/.config/autostart/netbook-launcher.desktop",
                          g_get_home_dir ());
      gnome_desktop_item_save (item, netbook_desktop, TRUE, &error);

      if (error)
        {
          g_warning ("Unable to replace ume-launcher.desktop with netbook-launcher.desktop: %s", error->message);

          g_error_free (error);
          g_free (ume_desktop);
          g_free (netbook_desktop);
          gnome_desktop_item_unref (item);
          return FALSE;
        }
      gnome_desktop_item_unref (item);

      /* Remove of ume_desktop file */
      g_unlink (ume_desktop);

      g_free (ume_desktop);
      g_free (netbook_desktop);

      return !autostart_enabled;
    }

  g_free (ume_desktop);
  return FALSE;
}
#if 0
static void
restart_nl ()
{
  gchar *exec;

  exec = g_strdup_printf ("netbook-launcher --no-restart");
  gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                    exec, NULL);
  g_free (exec);
  gtk_main_quit ();
}

static void
on_system_resume (NlConfig *config)
{
  if (!norestart)
    restart_nl ();
}
#endif
#if 0
/* What happens when the stage changes size
 * NOTE: Only send --no-restart if it's currently enabled, otherwise the
 * nl will not react when RESIZE>SUSPEND>RESUME
 */
static void
restart_nl_for_resize ()
{
  NlCatbar *bar;
  gchar *exec;

  bar = (NlCatbar *)nl_catbar_get_default ();
  exec = g_strdup_printf ("netbook-launcher --last-category=%d",
                          nl_catbar_get_active_category (bar));
  gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                    exec, NULL);
  g_free (exec);
  gtk_main_quit ();
}

static void
size_changed (GdkScreen    *screen,
              ClutterActor *app)
{
  static gboolean already_called = FALSE;

  if (CSW () == gdk_screen_get_width (screen)
      && CSH () == gdk_screen_get_height (screen))
    return;

  if (already_called)
    return;

  already_called = TRUE;
  restart_nl_for_resize ();
}

#endif
