/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-favorite-item.h"

#include <glib.h>
#include <glib/gi18n.h>
#include <clutk/clutk.h>
#include <launcher/launcher.h>
#include <netbook-launcher/netbook-launcher.h>
#include <clutter-gtk/clutter-gtk.h>
#include <libnotify/notify.h>

#include "nl-favorite-view.h"

G_DEFINE_TYPE (NlFavoriteItem, nl_favorite_item, G_TYPE_OBJECT);

#define NL_FAVORITES_ITEM_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE \
        ((obj), NL_TYPE_FAVORITES_ITEM, NlFavoriteItemPrivate))

struct _NlFavoriteItemPrivate
{
  NlShell           *shell;
  NlPixbufCache     *cache;
  LauncherFavorites *favorites;

  ClutterActor      *scroll;
  ClutterActor      *view;
  ClutterActor      *item;
};

enum
{
  PROP_0,
  PROP_SHELL
};

/* Forwards */
static NlFavorite * load_favorite       (NlFavoriteItem *self,
    const gchar    *uid);
static void on_favs_clicked     (ClutterActor   *actor,
                                 NlFavoriteItem *item);
static void on_favorite_added   (NlShell        *shell,
                                 const gchar    *uid,
                                 NlFavoriteItem *self);
static void on_favorite_removed (NlShell        *shell,
                                 const gchar    *uid,
                                 NlFavoriteItem *self);

/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  NlFavoriteItemPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITES_ITEM (object));
  priv = NL_FAVORITES_ITEM_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      priv->shell = g_value_get_pointer (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlFavoriteItemPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITES_ITEM (object));
  priv = NL_FAVORITES_ITEM_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      g_value_set_pointer (value, priv->shell);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}


static void
nl_favorite_item_finalize (GObject *object)
{
  NlFavoriteItemPrivate *priv;

  priv = NL_FAVORITES_ITEM_GET_PRIVATE (object);

  if (priv->favorites)
    {
      g_object_unref (priv->favorites);
      priv->favorites = NULL;
    }

  if (priv->cache)
    {
      g_object_unref (priv->cache);
      priv->cache = NULL;
    }

  if (priv->scroll)
    {
      g_object_unref (priv->scroll);
      priv->scroll = NULL;
    }

  G_OBJECT_CLASS (nl_favorite_item_parent_class)->finalize (object);
}

static void
on_content_view_child_changed (NlContentView  *view,
                               ClutterActor   *new_view,
                               NlFavoriteItem *item)
{
  g_return_if_fail (NL_IS_FAVORITES_ITEM (item));

  nl_sidebar_item_set_active (NL_SIDEBAR_ITEM (item->priv->item),
                              new_view == item->priv->scroll ? TRUE : FALSE);
}

static void
nl_favorite_item_constructed (GObject *object)
{
  NlFavoriteItemPrivate *priv = NL_FAVORITES_ITEM (object)->priv;
  ClutterActor     *item;
  GdkPixbuf        *pixbuf;
  ClutterContainer *container;

  priv->favorites = launcher_favorites_get_default ();

  container = (ClutterContainer *)nl_shell_get_sidebar (priv->shell);

  pixbuf = nl_pixbuf_cache_icon_for_name (priv->cache,
                                          "unr-applications-home",
                                          48);
  if (!pixbuf)
    pixbuf = nl_pixbuf_cache_icon_for_name (priv->cache,
                                          "emblem-favorite",
                                          48);

  item = nl_sidebar_item_new (_("Favorites"),
                              _("Your favorite applications, widgets "
                                "and files"),
                              pixbuf,
                              1);
  clutter_container_add_actor (container, item);
  clutter_actor_show (item);
  priv->item = item;

  g_signal_connect (item, "clicked", G_CALLBACK (on_favs_clicked), object);
  g_signal_connect (nl_shell_get_content_view (priv->shell), "changed",
                    G_CALLBACK (on_content_view_child_changed), object);

  g_object_unref (pixbuf);

  g_signal_connect (priv->shell, "favorite-added",
                    G_CALLBACK (on_favorite_added), object);
  g_signal_connect (priv->shell, "favorite-removed",
                    G_CALLBACK (on_favorite_removed), object);
}

static void
nl_favorite_item_class_init (NlFavoriteItemClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;

  obj_class->finalize     = nl_favorite_item_finalize;
  obj_class->constructed  = nl_favorite_item_constructed;
  obj_class->set_property = set_property;
  obj_class->get_property = get_property;

  /* Install properties */
  pspec = g_param_spec_pointer ("shell", "shell", "shell",
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_SHELL, pspec);

  g_type_class_add_private (obj_class, sizeof (NlFavoriteItemPrivate));
}

static gboolean
load_favorites (NlFavoriteItem *self)
{
  NlFavoriteItemPrivate *priv = self->priv;
  GSList *favs, *f;
  CtkPadding padding = { 0, 12, 0, 0 };
  
  priv->scroll = nl_scroll_view_new ();
  g_object_ref_sink (priv->scroll);
  ctk_actor_set_padding (CTK_ACTOR (priv->scroll), &padding);
  clutter_actor_show (priv->scroll);
  
  priv->view = nl_favorite_view_new (priv->shell);
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->scroll), priv->view);
  clutter_actor_show (priv->view);

  favs = launcher_favorites_get_favorites (priv->favorites);
  for (f = favs; f; f = f->next)
    {
      load_favorite (self, f->data);
    }
  g_slist_foreach (favs, (GFunc)g_free, NULL);
  g_slist_free (favs);

  g_signal_emit_by_name (priv->item, "clicked");

  return FALSE;
}

static void
nl_favorite_item_init (NlFavoriteItem *self)
{
  NlFavoriteItemPrivate *priv;

  priv = self->priv = NL_FAVORITES_ITEM_GET_PRIVATE (self);

  priv->cache = nl_pixbuf_cache_get_default ();

  g_idle_add ((GSourceFunc)load_favorites, self);
}

/*
 * Public methods
 */
NlFavoriteItem *
nl_favorite_item_new (NlShell *shell)

{
  NlFavoriteItem *favorite_item = NULL;

  favorite_item = g_object_new (NL_TYPE_FAVORITES_ITEM,
                                "shell", shell,
                                NULL);

  return favorite_item;
}

/*
 * Private methods
 */
static void
on_favs_clicked (ClutterActor *actor, NlFavoriteItem *item)
{
  NlFavoriteItemPrivate *priv;
  NlContentView *view;

  g_return_if_fail (NL_IS_FAVORITES_ITEM (item));
  priv = item->priv;

  view = nl_shell_get_content_view (NL_SHELL (priv->shell));
    
  if (nl_content_view_get_child (view) != priv->scroll)
    {
      nl_content_view_set_child (view, priv->scroll);
    }
}

static GdkPixbuf *
scale_pixbuf_preserve_aspect_ratio (GdkPixbuf *pixbuf,
                                    gint size)
{
  GdkPixbuf *new_icon;
  gint w, h, new_width, new_height, max_edge;

  w = gdk_pixbuf_get_width (pixbuf);
  h = gdk_pixbuf_get_height (pixbuf);
  max_edge = MAX (w, h);
  new_width = size * (w / max_edge);
  new_height = size * (h / max_edge);

  /* Scale the image down, preserving the aspect ratio */
  new_icon = gdk_pixbuf_scale_simple (pixbuf, new_width, new_height,
                                      GDK_INTERP_HYPER);

  return new_icon;
}

static gboolean
send_notification (const gchar *title,
                   const gchar *content,
                   GdkPixbuf   *pixbuf,
                   GtkWidget   *status_widget)
{
  gchar *name = NULL;
  GdkPixbuf *icon;
  gboolean free_icon = FALSE;
  NotifyNotification *notification;

  //notify_get_server_info (&name, NULL, NULL, NULL);

  if (!title || !content || !pixbuf)
    return FALSE;

  if (!g_strcmp0 ("notification-daemon", name))
    {
      icon = scale_pixbuf_preserve_aspect_ratio (pixbuf, 48);
      free_icon = TRUE;
    }
  else
    {
      icon = pixbuf;
    }

  notification = notify_notification_new (title, content, NULL, NULL);
  notify_notification_set_icon_from_pixbuf (notification, icon);

  notify_notification_show (notification, NULL);

  g_free (name);
  if (free_icon)
    {
      g_object_unref (G_OBJECT (icon));
    }

  return TRUE;
}

static NlFavorite *
load_favorite (NlFavoriteItem *self, const gchar *uid)
{
  NlFavoriteItemPrivate *priv;
  GSList                *loaders, *l;
  gchar                 *type;
  NlFavoritesLoader     *type_loader = NULL;
  NlFavorite            *fav = NULL;

  g_return_val_if_fail (NL_IS_FAVORITES_ITEM (self), NULL);
  priv = self->priv;

  type = launcher_favorites_get_string (priv->favorites, uid, "type");

  loaders = nl_shell_get_favorites_loaders (priv->shell);
  for (l = loaders; l; l = l->next)
    {
      NlFavoritesLoader *loader = l->data;

      if (nl_favorites_loader_handles_type (loader, type))
        {
          type_loader = loader;
          break;
        }
    }

  if (type_loader)
    {
      fav = nl_favorites_loader_load (type_loader, uid);

      if (!fav)
        {
          g_warning ("Cannot load favorite %s, of type %s", uid, type);
          g_free (type);
          return NULL;
        }
      nl_favorite_view_add_favorite (NL_FAVORITE_VIEW (priv->view), fav);
    }
  else
    {
      g_warning ("Unable to find FavoritesLoader for %s, type %s", uid, type);
    }

  g_free (type);

  return fav;
}

static gboolean
fade_out (NlFavoriteItem *self)
{
  g_return_val_if_fail (CTK_IS_ACTOR (self), FALSE);

  if (ctk_actor_get_state (CTK_ACTOR (self)) == CTK_STATE_PRELIGHT)
    {
      ctk_actor_set_state (CTK_ACTOR (self), CTK_STATE_NORMAL);
    }

  return FALSE;
}

static void
on_favorite_added   (NlShell        *shell,
                     const gchar    *uid,
                     NlFavoriteItem *self)
{
  NlFavorite *fav;

  fav = load_favorite (self, uid);
  if (fav)
    {
      send_notification (nl_favorite_get_name (fav), _("Added to favorites"),
                         nl_favorite_get_icon (fav), NULL);

      ctk_actor_set_state (CTK_ACTOR (self->priv->item),
                           CTK_STATE_PRELIGHT);
      g_timeout_add (200, (GSourceFunc)fade_out, self->priv->item);
    }
}

static void
on_favorite_removed (NlShell        *shell,
                     const gchar    *uid,
                     NlFavoriteItem *self)
{
  g_debug ("favorite removed: %s", uid);
}

