/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-places-manager.h"

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <clutk/clutk.h>
#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>
#include <netbook-launcher/netbook-launcher.h>

G_DEFINE_TYPE (NlPlacesManager, nl_places_manager, G_TYPE_OBJECT);

#define NL_PLACES_MANAGER_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  NL_TYPE_PLACES_MANAGER, \
  NlPlacesManagerPrivate))

struct _NlPlacesManagerPrivate
{
  NlShell *shell;
  ClutterActor  *item;
  ClutterActor  *scroll;
  ClutterActor  *vbox;
  ClutterActor  *bg_texture;
};

enum
{
  PROP_0,
  PROP_SHELL
};

/* Forwards */
static void on_places_clicked (NlSidebarItem   *item,
                               NlPlacesManager *manager);

/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  NlPlacesManagerPrivate *priv;

  g_return_if_fail (NL_IS_PLACES_MANAGER (object));
  priv = NL_PLACES_MANAGER_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      priv->shell = g_value_get_pointer (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlPlacesManagerPrivate *priv;

  g_return_if_fail (NL_IS_PLACES_MANAGER (object));
  priv = NL_PLACES_MANAGER_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      g_value_set_pointer (value, priv->shell);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}
static void
nl_places_manager_finalize (GObject *object)
{
  NlPlacesManagerPrivate *priv;

  priv = NL_PLACES_MANAGER_GET_PRIVATE (object);

  if (priv->vbox)
    {
      clutter_actor_destroy (priv->vbox);
      priv->vbox = NULL;
    }

  if (priv->bg_texture)
    {
      clutter_actor_destroy (priv->bg_texture);
      priv->bg_texture = NULL;
    }

  G_OBJECT_CLASS (nl_places_manager_parent_class)->finalize (object);
}

static void
on_content_view_child_changed (NlContentView   *content_view,
                               ClutterActor    *new_view,
                               NlPlacesManager *manager)
{
  g_return_if_fail (NL_IS_PLACES_MANAGER (manager));

  if (new_view == manager->priv->scroll)
    nl_sidebar_item_set_active (NL_SIDEBAR_ITEM (manager->priv->item), TRUE);
  else
    nl_sidebar_item_set_active (NL_SIDEBAR_ITEM (manager->priv->item), FALSE);
}

static void
nl_places_manager_constructed (GObject *object)
{
  NlPlacesManagerPrivate *priv;
  GdkPixbuf       *pixbuf;
  ClutterActor    *item;
  NlSidebar       *sidebar;
  NlPixbufCache   *cache;

  priv = NL_PLACES_MANAGER_GET_PRIVATE (object);

  /* Add the Places section to the sidebar */
  sidebar = nl_shell_get_sidebar (priv->shell);
  cache = nl_pixbuf_cache_get_default ();

  pixbuf = nl_pixbuf_cache_icon_for_name (cache,
                                          "unr-applications-places",
                                          48);
  if (!pixbuf)
    pixbuf = nl_pixbuf_cache_icon_for_name (cache,
                                            "folder",
                                            48);

  item = priv->item = nl_sidebar_item_new (_("Files & Folders"),
                      _("Files & Folders"),
                      pixbuf,
                      100);
  clutter_container_add_actor (CLUTTER_CONTAINER (sidebar), item);
  g_signal_connect (item, "clicked", G_CALLBACK (on_places_clicked), object);
  g_object_unref (pixbuf);

  g_signal_connect (nl_shell_get_content_view (priv->shell), "changed",
                    G_CALLBACK (on_content_view_child_changed), object);

  /* Scrolled view */
  priv->scroll = nl_scroll_view_new ();
  g_object_ref_sink (priv->scroll);

  /* VBox to hold the sources */
  priv->vbox = ctk_vbox_new (12);
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->scroll), priv->vbox);
  clutter_actor_show (priv->vbox);
}

static void
nl_places_manager_class_init (NlPlacesManagerClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;

  obj_class->finalize     = nl_places_manager_finalize;
  obj_class->constructed  = nl_places_manager_constructed;
  obj_class->set_property = set_property;
  obj_class->get_property = get_property;

  pspec = g_param_spec_pointer ("shell", "shell", "shell",
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_SHELL, pspec);

  g_type_class_add_private (obj_class, sizeof (NlPlacesManagerPrivate));
}

static void
nl_places_manager_init (NlPlacesManager *manager)
{
  NlPlacesManagerPrivate *priv;

  priv = manager->priv = NL_PLACES_MANAGER_GET_PRIVATE (manager);

  priv->bg_texture = clutter_texture_new_from_file (
                       PKGDATADIR"/iconview_normal.png", NULL);
}

/*
 * Public methods
 */
NlPlacesManager *
nl_places_manager_new (NlShell *shell)

{
  NlPlacesManager *places_manager;

  g_return_val_if_fail (NL_IS_SHELL (shell), NULL);

  places_manager = g_object_new (NL_TYPE_PLACES_MANAGER,
                                 "shell", shell,
                                 NULL);

  return places_manager;
}

/*
 * Private methods
 */
static void
on_places_clicked (NlSidebarItem   *item,
                   NlPlacesManager *manager)
{
  NlPlacesManagerPrivate *priv;
  NlContentView *view;

  g_return_if_fail (NL_IS_PLACES_MANAGER (manager));
  priv = manager->priv;

  view = nl_shell_get_content_view (NL_SHELL (priv->shell));

  if (nl_content_view_get_child (view) != priv->scroll)
    nl_content_view_set_child (view, priv->scroll);
}

/*
 * Public Methods
 */
void
nl_places_manager_append_source (NlPlacesManager *manager,
                                 ClutterActor    *source)
{
  NlPlacesManagerPrivate *priv;
  ClutterActor           *bg;

  g_return_if_fail (NL_IS_PLACES_MANAGER (manager));
  g_return_if_fail (CLUTTER_IS_ACTOR (source));
  priv = manager->priv;

  if (CTK_IS_ACTOR (source))
    {
      CtkPadding padding = { 15, 15, 15, 15 };

      bg = nl_texture_frame_new (CLUTTER_TEXTURE (priv->bg_texture),
                                 25, 25, 25, 25);

      ctk_actor_set_background_for_state (CTK_ACTOR (source), CTK_STATE_NORMAL,
                                          bg);
      ctk_actor_set_padding (CTK_ACTOR (source), &padding);
    }

  ctk_box_pack (CTK_BOX (priv->vbox), source, FALSE, FALSE);
}

