/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/

#include <linux/interrupt.h>
#include <linux/debugfs.h>
#include <linux/kthread.h>
#include <linux/slab.h>

#include "iwl-idi.h"
#include "trans_slave/idi_al.h"
#include "iwl-emulation.h"
#include "iwl-amfh.h"

/*
 * Logical channels define flags:
 */

/* IDI Defines */
#define DEBUGFS_FILE_NAME		"em_idi_status"
#define IDI_AL_INTER			0
#define IDI_DBB_INTER			1

/*
 * IWL IDI Emulation Logger
 */
#define IDI_LOG_PREFIX			"[IDI]"

#ifdef __IWL_IDI_LOG_ENABLED__
#define IDI_TRACE_ENTER \
	IWL_EM_TRACE_ENTER(__IDI_LOG_LEVEL_TRACE__, IDI_LOG_PREFIX)
#define IDI_TRACE_EXIT \
	IWL_EM_TRACE_EXIT(__IDI_LOG_LEVEL_TRACE__, IDI_LOG_PREFIX)
#define IDI_TRACE_EXIT_RET_STR(_ret) \
	IWL_EM_LOG(__IDI_LOG_LEVEL_TRACE__, IDI_LOG_PREFIX, "<<< "_ret)
#define IDI_TRACE_EXIT_RET(_ret) \
	IWL_EM_TRACE_EXIT_RET(__IDI_LOG_LEVEL_TRACE__, IDI_LOG_PREFIX, _ret)
#define IWL_IDI_LOG(fmt, args ...) \
	IWL_EM_LOG(__IDI_LOG_LEVEL_DEBUG__, IDI_LOG_PREFIX, fmt, ## args)
#define IWL_IDI_LOG_HEX_DUMP(msg, p, len) \
	IWL_EM_LOG_HEX_DUMP(msg, p, len)
#else

#define IDI_TRACE_ENTER
#define IDI_TRACE_EXIT
#define IDI_TRACE_EXIT_RET_STR(_ret)
#define IDI_TRACE_EXIT_RET(_ret)
#define IWL_IDI_LOG(fmt, args ...)
#define IWL_IDI_LOG_HEX_DUMP(msg, p, len)

#endif
/* Always print error messages */
#define IWL_IDI_LOG_ERR(fmt, args ...) \
	IWL_EM_LOG_ERR(fmt, ## args)

/*
 * This struct represents a logical channel in the IDI HW emulation.
 *
 * @channel - Logical channel number.
 * @irq_handler[IWL_IDI_NUM_DIR] - The handler function for the irq that will
 *		be called when the Interrupt bit in the sg list link is up.
 * @data: A void pointer to data that will be sent to the interrupt handler
 * when it is called.
 * @dma_enabled[IWL_IDI_NUM_DIR] - DMA enabled flag,  one for each direction.
 * @sg_ll[IWL_IDI_NUM_DIR] - The DMA SG list for the DBB channel,
 *						one for each direction.
 * @addr - The address in the ABB side.
 */
struct iwl_idi_dma_channel {
	void			*irq_handler;
	void			*irq_data;
	atomic_t		dma_enabled;
	void			*sg_ll;
	u32			addr;
};

/*
 * Represents the IDI DMA Struct for the IDI HW emulation.
 */
struct iwl_idi_t {

	/* DBB to AL DMA logical Channels
	 *  There is a struct for each channel in the DBB to AL */
	struct iwl_idi_dma_channel dbb_channels[IWL_IDI_MAX_CHANNELS];

	/* Tasklets:
	 * A tasklet is schedualled when an interrupt is called from the IDI
	 * to the host CPU - This call is done when the interrupt bit is on in
	 * the sg list.*/
	struct tasklet_struct dbb_tasklet[IWL_IDI_MAX_CHANNELS];

	/* Tasklets:
	 * A tasklet is schedualled when an interrupt is called from the IDI
	 * in the DBB side to the host CPU -Not the interrupt called in
	 * the sg list.*/
	struct tasklet_struct idi_tasklet;

	/* Used by the IDI to keep track if there is work schedualled */
	bool dma_work;

	/* DMA Thread for the DMA jobs*/
	struct task_struct *idi_dma_em;

	/* Used to signal that the DBB has requested that the rx will
	 *  be stopped */
	bool idi_dbb_rx_enabled;

	/* Export DMA data to debugfs */
	struct dentry *dma_debugfs_file;

};

/* Global IDI struct */
static struct iwl_idi_t iwl_idi_em_global;
struct iwl_idi_t *iwl_idi_em = &iwl_idi_em_global;

/* Forward declerations: */
struct idi_sg_desc;
static void iwl_idi_dbb_inter_disp(unsigned long data);
static int iwl_idi_add_debugfs(void);

/**
 * Initialize the idi emulation block.
 *
 * @idi: reference to struct iwl_idi_t
 * @a5m: reference to the A5M CPU that the idi connect to.
 *
 * Returns 0 on success,  negative value describing the error otherwise.
 */
int iwl_idi_init(void)
{
	u8 channel, ret_error;

	IDI_TRACE_ENTER;
	IWL_IDI_LOG("Initiating ABB and AL logical channels");

	/* Clean IDI */
	memset(iwl_idi_em, 0, sizeof(struct iwl_idi_t));

	/* Init the DBB to AL logical channel */
	iwl_idi_em->dbb_channels[0].addr = IWL_IDI_ABB_TX_HP_ADDR;
	iwl_idi_em->dbb_channels[1].addr = IWL_IDI_ABB_TX_LP_ADDR;
	iwl_idi_em->dbb_channels[2].addr = IWL_IDI_ABB_RX_ADDR;

	/* Init the DBB channels dma flag. */
	for (channel = 0; channel < (IWL_IDI_MAX_CHANNELS); channel++) {
		atomic_set(&(iwl_idi_em->dbb_channels[channel].dma_enabled),
			   IWL_IDI_DISABLED);
	}
	IWL_IDI_LOG("Logical channels initialized");

	/* DBB Tasklet init */
	for (channel = 0; channel < (IWL_IDI_MAX_CHANNELS); channel++)
		tasklet_init(&(iwl_idi_em->dbb_tasklet[channel]),
			     iwl_idi_dbb_inter_disp, channel);
	IWL_IDI_LOG("DBB Tasklets initialized");

	/* AL Tasklet init */
	tasklet_init(&(iwl_idi_em->idi_tasklet), NULL, 0);
	tasklet_disable(&(iwl_idi_em->idi_tasklet));
	IWL_IDI_LOG("Tasklet initialized and disabled");

	/* Init the IDI RX engine */
	ret_error = iwl_idi_rx_em_init();
	if (ret_error) {
		IWL_IDI_LOG_ERR("Cannot Initialize IDI RX");
		goto exit;
	}
	IWL_IDI_LOG("Initializing the IDI RX - Done.");

	/* Init the IDI TX engine */
	ret_error = iwl_idi_tx_em_init();
	if (ret_error) {
		IWL_IDI_LOG_ERR("Cannot Initialize IDI TX");
		goto free_idi_rx;
	}

	/* Init dma debugfs */
	iwl_idi_em->dma_debugfs_file = NULL;

	IWL_IDI_LOG("DMA emulator thread initialized");
	goto exit;

free_idi_rx:
	iwl_idi_rx_em_free();
exit:
	IDI_TRACE_EXIT;
	return ret_error;
}

/**
 * Free the idi emulation block. The API should be called only after all its
 * operation were gracefully stopped. This mean that the free will fail in
 * case that one of the channel is enabled to transmit data.
 *
 *  @idi:reference to struct iwl_idi_t
 */
void iwl_idi_free(void)
{
	u8 channel = 0, running = 0;
	struct iwl_idi_dma_channel *dbb_ch = NULL;
	IDI_TRACE_ENTER;

	/* Go over the channels and check if any one of them
	 * is enabled while trying to Release the AL IDI dma HW
	 */
	for (channel = 0; channel < IWL_IDI_MAX_CHANNELS; channel++) {
		dbb_ch = &(iwl_idi_em->dbb_channels[channel]);
		if (IWL_IDI_ENABLED == atomic_read(&dbb_ch->dma_enabled)) {
			IWL_IDI_LOG_ERR(
				"IDI_Free_ Called while still running DMA TX,Releasing DBB DMA channel %d",
				channel);
			running = IWL_IDI_ENABLED;
		}
	}

	if (running)
		IWL_IDI_LOG_ERR("DMA tasks are ENABLED while trying to free");
	else
		IWL_IDI_LOG("All DMA tasks are disabled");

	iwl_idi_rx_em_free();
	IWL_IDI_LOG("Freed IDI RX");

	iwl_idi_tx_em_free();
	IWL_IDI_LOG("Freed IDI TX");

	IWL_IDI_LOG("Released IDI");
	IDI_TRACE_EXIT;
}

/**
 * Start the IDI Emulation.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_start(void)
{
	u32 ret_error;

	IDI_TRACE_ENTER;

	/* Signal that the idi dbb rx dma is enabled */
	iwl_idi_em->idi_dbb_rx_enabled = true;
	IWL_IDI_LOG("Start IDI RX - Done.");

	/* Start the IDI RX */
	ret_error = iwl_idi_rx_em_start();
	if (ret_error) {
		IWL_IDI_LOG_ERR("Starting IDI RX Failed");
		goto exit;
	}

	/* Start the IDI TX */
	ret_error = iwl_idi_tx_em_start();
	if (ret_error) {
		IWL_IDI_LOG_ERR("Starting IDI TX Failed");
		goto stop_idi_rx;
	}

	/* Debugfs INIT
	  * This is done on start since there is still no valid pointer to the
	  * ieee80211 debugfs dentry */
	if (iwl_idi_add_debugfs())
		IWL_IDI_LOG("Debugfs error, NOT initialized");
	else
		IWL_IDI_LOG("Debugfs initialized");

	IWL_IDI_LOG("IDI Emulation started");
	goto exit;

stop_idi_rx:
	iwl_idi_rx_em_stop();
exit:
	IDI_TRACE_EXIT;
	return ret_error;
}

/**
 * Stop the IDI Emulation
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_stop(void)
{
	int channel;

	IDI_TRACE_ENTER;

	/* Signal that the idi dbb rx dma is enabled */
	iwl_idi_em->idi_dbb_rx_enabled = false;

	/* Stop and kill tasklets */
	for (channel = 0; channel < (IWL_IDI_MAX_CHANNELS); channel++)
		tasklet_kill(&(iwl_idi_em->dbb_tasklet[channel]));
	tasklet_kill(&(iwl_idi_em->idi_tasklet));

	/* Stop the IDI RX */
	if (iwl_idi_rx_em_stop())
		IWL_IDI_LOG_ERR("Stopping IDI RX - FAILED");

	/* Stop the IDI TX */
	if (iwl_idi_tx_em_stop())
		IWL_IDI_LOG_ERR("Stopping IDI TX - FAILED");

	IWL_IDI_LOG("IDI Emulation stopped");
	IDI_TRACE_EXIT;
	return 0;
}

/*
 * Returns the state of the RX engine in the DMA.
 *
 * Returns true if the rx dma engine is enabled, false otherwise.
 */
int iwl_idi_rx_status_enabled(void)
{
	return iwl_idi_em->idi_dbb_rx_enabled;
}

/*
 * Test that the given channel and direction corresponds to the
 * channels and directions available.
 *
 * @chan - A channel number to check.
 * @dir - A direction to check.
 * Returns 0 on success,  negative value describing the error otherwise.
 */
static inline int iwl_idi_sanity_channel(u32 chan, u32 dir)
{
	IDI_TRACE_ENTER;

	/* Sanity Channel */
	if ((chan > IWL_IDI_MAX_CHANNELS) ||
	    chan < IWL_IDI_CHANNEL_START) {
		IWL_IDI_LOG_ERR("Channel Number error, Given %d", chan);
		return -EINVCHAN;
	}

	/* Sanity Direction */
	if ((dir != IWL_IDI_DBB_TX) && (dir != IWL_IDI_DBB_RX))
		IWL_IDI_LOG_ERR("Direction error,  Given %d", dir);

	IDI_TRACE_EXIT;
	return 0;
}

/**
 *  Set the interrupt handler for the given channel, in the given direction
 *  The call should be done by the driver transport layer.
 *
 * @chan: the channel
 * @dir: the direction,  IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 * @irq_handler: the function pointer to the irq_handler
 * @data: A void pointer to data that will be sent to the interrupt handler
 * when it is called.
 * Returns 0 on success,  negative value describing the error otherwise.
 */
int iwl_idi_dbb_set_irq_handler(u32 chan,
				u32 dir,
				iwl_idi_dbb_dma_irq irq_handler,
				void *data)
{
	IDI_TRACE_ENTER;

	/* Sanity checks */
	if (iwl_idi_sanity_channel(chan, dir))
		return -EINVCHAN;

	/* Store the irq handler */
	iwl_idi_em->dbb_channels[chan].irq_handler = irq_handler;
	iwl_idi_em->dbb_channels[chan].irq_data = data;
	IWL_IDI_LOG("IRQ Handler for Channel %d,  Direction %d - Registered ",
		    chan, dir);
	IDI_TRACE_EXIT;
	return 0;
}

/*
 * AL converters from the dma descriptor to the requested field
 */
static inline u32 iwl_idi_calc_al_size(struct iwl_al_dma_desc *sg_ll)
{
	return (sg_ll->size >> IDI_SIZE_OFFSET) & IDI_SIZE_MASK;
}

static inline unsigned long iwl_idi_calc_al_addr_src(
						struct iwl_al_dma_desc *sg_ll)
{
	return sg_ll->src;
}

static inline unsigned long iwl_idi_calc_al_addr_dst(
						struct iwl_al_dma_desc *sg_ll)
{
	return sg_ll->dst;
}

static inline u32 iwl_idi_calc_al_I_bit(struct iwl_al_dma_desc *sg_ll)
{
	return (sg_ll->size) & IDI_SGLIST_I_BIT;
}

static inline u32 iwl_idi_calc_al_EL_bit(struct iwl_al_dma_desc *sg_ll)
{
	return (sg_ll->size) & IDI_SGLIST_L_BIT;
}

static inline void *iwl_idi_calc_al_next(struct iwl_al_dma_desc *sg_ll)
{
	if (!iwl_idi_calc_al_EL_bit(sg_ll))
		return sg_ll + 1;

	IWL_IDI_LOG("EL bit is on in AL");
	return NULL;
}

/*
* Calls the dbb IRQ handler with the DBB interrupt data.
* Called in tasklet context.
*
*@data: The DBB interrupt data casted to unsigned long.
*/
static void iwl_idi_dbb_inter_disp(unsigned long data)
{
	u32 channel = (u32)data;
	__le32 dir;
	void *irq_data = iwl_idi_em->dbb_channels[channel].irq_data;

	IDI_TRACE_ENTER;

	/* Set DBB direction accroding to channel */
	if (IWL_IDI_RX_CHANNEL == channel)
		dir = cpu_to_le32(IWL_IDI_DBB_RX);
	else
		dir = cpu_to_le32(IWL_IDI_DBB_TX);

	/* Call the DBB irq handler with the required channel and direction */
	((iwl_idi_dbb_dma_irq)
		iwl_idi_em->dbb_channels[channel].irq_handler)
						(cpu_to_le32(channel),
						 dir,
						 irq_data);

	IWL_IDI_LOG("DBB interrupt tasklet done");
	IDI_TRACE_EXIT;
}

/*
 * Calls the interrupt that was registered for the given channel in the given
 * direction and to the required module(ABB/DBB).
 *
 * @channel - The channel number that the interrupt was called in.
 * @dir - The direction that the interrupt was called in.
 */
void iwl_idi_call_inter(u32 channel)
{
	IDI_TRACE_ENTER;

	/* Sanity check */
	if (NULL == iwl_idi_em->dbb_channels[channel].irq_handler) {
		IWL_IDI_LOG_ERR("DBB IRQ handler called is null in cannel %d",
				channel);
		return;
	}

	/* Schedule tasklet
	 * In case that there are more then 2 interrupts on the same channel
	 * the interrupts will be masked - same as HW behaviour */
	tasklet_schedule(&iwl_idi_em->dbb_tasklet[channel]);

	IDI_TRACE_EXIT;
}

/**
 *  Set a scatter/gather list in the given channel, in the given direction.
 *  The call should be done only when the <chan,dir> is stopped.
 *
 *  @chan: the channel
 *  @dir: the direction,  IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *  @sg_ll: scatter/gather list
 *
 *  Returns 0 on success,  negative value describing the error otherwise.
 */
int iwl_idi_dbb_set_sg(u32 chan, u32 dir,
		       struct idi_sg_desc *sg_ll)
{
	IDI_TRACE_ENTER;

	/* Channel Sanity checks */
	if (iwl_idi_sanity_channel(chan, dir))
		return -EINVCHAN;
	WARN_ON(NULL == sg_ll);

	IWL_IDI_LOG("Setting SG List for Channel %d, Direction %d",
		    chan, dir);

	/* Check if the DMA is enabled while changing the sg list */
	if (atomic_read(&(iwl_idi_em->dbb_channels[chan].dma_enabled))
							== IWL_IDI_ENABLED) {
		IWL_IDI_LOG_ERR(
			"Setting SG list while dma is enabled for channel %d,   direction %d",
			chan, dir);
		return -EDMAENABLED;
	}

	/* Store the S/G List */
	iwl_idi_em->dbb_channels[chan].sg_ll = sg_ll;

	IDI_TRACE_EXIT;
	return 0;

}

/**
 * Start the DBB DMA operation on the given channel, in the given direction.
 * The DMA operation should be handled asyncronuously. A setting of the sg
 * list must be done before calling this API
 *
 *  @chan: the channel
 *  @dir: the direction,  IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *
 *  Returns 0 on success,  negative value describing the error otherwise.
 */
int iwl_idi_dbb_start(u32 chan, u32 dir)
{
	int ret = 0;
	IDI_TRACE_ENTER;

	/* Channel Sanity checks */
	if (iwl_idi_sanity_channel(chan, dir))
		return -EINVCHAN;

	/* DMA Sanity check */
	if (atomic_read(&(iwl_idi_em->dbb_channels[chan].dma_enabled))
	    == IWL_IDI_ENABLED) {
		IWL_IDI_LOG_ERR(
			"Starting the DMA while it's already started %d",
			chan);
		return -EDMAENABLED;
	}

	/* Enable the DMA flag */
	IWL_IDI_LOG("Enabled DMA on channel %d dir %d", chan, dir);
	atomic_set(&(iwl_idi_em->dbb_channels[chan].dma_enabled),
		   IWL_IDI_ENABLED);

	/* Direct traffic to the correct flow */
	if (IWL_IDI_RX_CHANNEL != chan) {
		iwl_idi_tx_em_dma_enable(chan);
	} else {
		iwl_idi_em->idi_dbb_rx_enabled = true;
		iwl_idi_rx_em_alert_change();
	}

	IDI_TRACE_EXIT;
	return ret;
}

/**
 *  Stop the DBB DMA in the given channel,  in the given direction.
 *  The call is syncronuous,  and on return the DMA is halted.
 *
 *  @chan: the channel
 *  @dir: the direction,  IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *
 *  Returns 0 on success,  negative value describing the error otherwise.
 */
int iwl_idi_dbb_stop(u32 chan, u32 dir)
{

	int ret = 0;

	IDI_TRACE_ENTER;

	/* Channel Sanity checks */
	if (iwl_idi_sanity_channel(chan, dir)) {
		ret = -EINVCHAN;
		goto exit;
	}

	/* DMA Sanity check */
	if (atomic_read(&(iwl_idi_em->dbb_channels[chan].dma_enabled))
							== IWL_IDI_DISABLED) {
		IWL_IDI_LOG(
			"Stopping the DMA while it's already stopped on channel %d",
			chan);
		goto exit;
	}

	/* Disable the dma flag */
	atomic_set(&(iwl_idi_em->dbb_channels[chan].dma_enabled),
		   IWL_IDI_DISABLED);
	IWL_IDI_LOG("Stopping DBB DMA for Channel %d, Direction %d",
		    chan, dir);

	/* Wait until the transaction has ended before returning */
	if (IWL_IDI_RX_CHANNEL == chan) {
			iwl_idi_em->idi_dbb_rx_enabled = false;
			iwl_idi_rx_em_alert_change();
			iwl_idi_rx_em_wait_for_rx_stop();
	}

	/* Clear the SG list */
	iwl_idi_em->dbb_channels[chan].sg_ll = NULL;

	IWL_IDI_LOG("DBB DMA for Channel %d,  Direction %d - STOPPED",
		    chan, dir);

exit:
	IDI_TRACE_EXIT;
	return ret;
}

/*
 * Print the IDI error data.
 * This will be called when the emulation error data is called to be
 * printed.
 */
void iwl_em_idi_print_error_data(void)
{
}

/*****************************************************************************
 *
 * Debugfs Framework functions
 *
 ****************************************************************************/

#ifdef CPTCFG_IWLWIFI_DEBUGFS

static ssize_t iwl_dbgfs_idi_em_read(struct file *file,
				char __user *user_buf,
				size_t count, loff_t *ppos);

static const struct file_operations iwl_dbgfs_idi_em_ops = {
	.read = iwl_dbgfs_idi_em_read,
	.open = simple_open,
	.llseek = generic_file_llseek,
};

/* create files */
static int iwl_idi_add_debugfs(void)
{
	struct dentry *em_debugfs_dir = iwl_emulation_get_debugfs_dir();

	/*
	 * If the emulation debugfs directory doesn't exist yet
	 * or the file for the idi dma has already been created,
	 * abort
	 */
	if ((NULL == em_debugfs_dir) || iwl_idi_em->dma_debugfs_file)
		return 0;

	iwl_idi_em->dma_debugfs_file =
		debugfs_create_file(DEBUGFS_FILE_NAME,
				    S_IRUSR,
				    em_debugfs_dir,
				    iwl_al_em,
				    &iwl_dbgfs_idi_em_ops);

	if (ERR_PTR(-ENODEV) == iwl_idi_em->dma_debugfs_file) {
		IWL_IDI_LOG("Cannot crate debugfs file");
		iwl_idi_em->dma_debugfs_file = NULL;
		return -ENOMEM;
	}

	return 0;
}

/*
 * Prints all of the SG lists which are stored in the DBB channels into
 * the proc file buffer.
 *
 *@sg_ll - Reference to the first sg list head.
 *@buffer - The buffer to write to.
 *@size - reference to the size that returns as the size of the print buffer.
 */
static void iwl_idi_print_sg_ll_dbb(struct idi_sg_desc *sg_ll,
				    char *buffer, int *size)
{
	*size += sprintf(buffer + *size, "next \taddr \tsize\n");
	*size += sprintf(buffer + *size, "---- \t---- \t----\n");

	if (NULL == sg_ll) {
		*size += sprintf(buffer + *size,
				 "--- DBB S/G list is null ---\n");
	} else {
		do {
			*size += sprintf(buffer + *size, "%X \t%X \t%X\n",
					 sg_ll->next, sg_ll->base, sg_ll->size);
		} while (NULL != iwl_idi_calc_dbb_next(sg_ll));
	}
}

/**
 * This function is called on every proc file read.
 * The output on the buffer will be the entire data stored in the IDI struct.
 */
static ssize_t iwl_dbgfs_idi_em_read(struct file *file,
					char __user *user_buf,
					size_t count, loff_t *ppos){
	int pos = 0;
	char *buffer;
	u8 channel = 0;
	int ret = 0;
	int buf_size = 2048;

	buffer = kzalloc(buf_size, GFP_KERNEL);
	if (!buffer)
		return -ENOMEM;

	pos = sprintf(buffer, "IWL_IDI Emulation:\n\n");

	pos += sprintf(buffer + pos, "DBB IRQ: %s and %s",
		       (atomic_read(&(iwl_idi_em->idi_tasklet.count))) ?
							"Enabled" : "DIsabled",
		       (iwl_idi_em->idi_tasklet.state) ?
						"Scheduled" : "NOT Scheduled");

	/* Print the DBB channels data */
	pos += sprintf(buffer + pos, "****** DBB Channels ******\n");
	for (channel = 0; channel < (IWL_IDI_MAX_CHANNELS); channel++) {
		pos += sprintf(buffer + pos, "############################\n");
		pos += sprintf(buffer + pos, "Channel =  %d\n", channel);
		pos += sprintf(buffer + pos, "TX irq handler =  0x%lu\n",
			       (unsigned long)
			       iwl_idi_em->dbb_channels[channel].irq_handler);
		pos += sprintf(buffer + pos, "DMA %s",
			       (atomic_read(&(iwl_idi_em->dbb_channels[channel]
			       .dma_enabled))) ?
			       "Enabled" : "TX DMA Disabled\n");

		pos += sprintf(buffer + pos, "TX S/G List\n");
		iwl_idi_print_sg_ll_dbb(iwl_idi_em->dbb_channels[channel].sg_ll
					, buffer, &pos);
	}

	pos += sprintf(buffer + pos, "Debug print done\n");

	ret = simple_read_from_buffer(user_buf, count, ppos, buffer, pos);
	kfree(buffer);

	return ret;
}

#else
static int iwl_idi_add_debugfs(void)
{
	return 0;
}

#endif

/*****************************************************************************
 *
 * Logical channel fields Setters and Getters
 *
 ****************************************************************************/

/*
 * Returns the DBB SG list according to given channel.
 */
struct idi_sg_desc *iwl_idi_get_sg_ll(u32 channel)
{
	return iwl_idi_em->dbb_channels[channel].sg_ll;
}

/*
 * Sets the SG linked list in the DBB channelsaccording to
 * the given value.
 */
void iwl_idi_set_sg_ll(u32 channel, void *value)
{
	iwl_idi_em->dbb_channels[channel].sg_ll = value;
}

/*
 * Returns the DBB dma channel status pointer according to given channel.
 */
atomic_t *iwl_idi_get_dma_state(u32 channel)
{
	return &(iwl_idi_em->dbb_channels[channel].dma_enabled);
}

