#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import evas.decorators
import edje
import edje.decorators
import layout
import base
from efl_utils.animations import DecelerateTimelineAnimation as TimelineAnimation
from efl_utils.callbacks import generate_multiple_wait


__all__ = ("MainWindow", "StatusIcon")


class StatusIcon(base.EdjeWidget):
    """Status icon visual representation."""
    def __init__(self, main_window, icon, theme=None):
        base.EdjeWidget.__init__(self, main_window.evas,
                                 "icon/main_window_status/" + icon,
                                 main_window, theme)
        main_window.add_status_icon(self)
        self.callback_theme_changed = None
        self.show()

    def theme_changed(self):
        if self.callback_theme_changed is not None:
            self.callback_theme_changed(self)

    def delete(self):
        self._parent_widget.del_status_icon(self)
        base.EdjeWidget.delete(self)


class NotifyWindow(base.EdjeWidget):
    def __init__(self, main_window, group, message, theme=None):
        base.EdjeWidget.__init__(self, main_window.evas,
                                 "notify/" + group,
                                 main_window, theme)
        self.message = message

    def message_get(self):
        return self._message

    def message_set(self, message):
        self._message = message
        self.part_text_set("notify_text", message)

    message = property(message_get, message_set)


class MainWindow(base.EdjeWidget):
    """Terra's main window view.

    Introduction
    ============

    It acts as a B{window manager} view, the "windows" being L{screen.Screen}
    subclasses, and provides status icon area, "back" and "options" buttons,
    title and also a button to allow changing the current screen.


    Title Bar
    =========

    Title bar contains the B{status icons area}, current screen B{title} and
    also a button to allow changing the current screen, called B{"multitask"}.

    Title are I{advised} to be set only by screens, that can call L{title_set()}
    or L{title_replace()} if animation is desired.

    In order to control it's visibility, you can use L{titlebar_hide()} and
    L{titlebar_show()}.


    Status Icons
    ============

    Instances of L{StatusIcon} will be laid out horizontally in a status
    icons area, defined in Edje by the part called B{"titlebar:icons"}.

    Visibility of status icons area is tied to B{"titlebar"} visibility.


    Controls
    ========

    This is composed by buttons present on every screen: B{back} and B{options},
    the latter being shown or hidden depending on the return value of Screen
    method C{has_options()}.

     - clicking "back" will call C{self.callback_back(self)}, which must be set
       by the controller. It's the same as calling L{back()} method.
     - clicking "options" will call current screen's C{options(self)}.

    Hardware keys are also supported:
     - B{Escape:} same as L{back()};
     - B{F6, f:} same as L{toggle_fullscreen()}.

    Visibility can be set with L{controls_hide()} and L{controls_show()}.


    Panels
    ======

    Often used to present options and settings, they're stacked and while
    visible the underlying screen gets no events other than back.


    Screen Contents
    ===============

    Instances of L{screen.Screen} will be swallowed in current Edje at part
    B{"contents"}. These should be set using L{use()} or L{transition_to()}
    methods.

    L{use()} method just replace (maybe with minor animations) the existing
    screen with a new one, while L{transition_to()} will call
    L{screen.Screen.transition_to()} on the current screen and
    L{screen.Screen.transition_from()} on the new screen, waiting for both
    transitions to end and then calls C{end_callback()} to notify it finished.

    Screen Contents gain focus when swallowed, so they can intercept key
    events. They can call Window's L{handle_key_up()} and L{handle_key_down()}
    to forward this events. L{screen.Screen} has a proper way to repassing
    this events, by just overriding certain methods and return True when
    repass is desired.

    """

    stack_panel_timeout = 0.6
    stack_panel_offset = 30

    (MULTITASK_NONE,
     MULTITASK_PLAYING,
     MULTITASK_PAUSED) = xrange(3)

    multitask_state_signals = ("reset", "nowplaying_on", "nowpaused_on")

    def __init__(self, canvas, theme):
        self._current_screen = None
        self._box_stat_icons = None
        self.callback_back = None
        self.callback_toggle_fullscreen = None
        self.callback_multitask_clicked = None
        self.callback_go_home = None
        self.callback_panel_back = None
        self._anim_title = None
        self._titlebar_visible = True
        self._btn_back_visible = True
        self._btn_options_visible = True
        self._multitask_state = self.MULTITASK_NONE
        self._panel_stack = []
        self._notify = None
        base.EdjeWidget.__init__(self, canvas, "main", theme=theme)

        self._box_stat_icons = layout.HBox(self.evas, hpadding=5)
        self._box_stat_icons.show()
        self.part_swallow("titlebar:icons", self._box_stat_icons)

        self._cur_btn_back = None
        self._btn_back = base.EdjeWidget(canvas, "bt_back_default", self)

        self._cur_btn_options = None
        self._btn_options = base.EdjeWidget(canvas, "bt_options_default", self)

        self.control_buttons_set(self._btn_back, self._btn_options)

        self.key_up_bindings = {}
        self.key_down_bindings = {}

    @evas.decorators.resize_callback
    def _cb_on_resize(self):
        w, h = self.size_get()
        n = len(self._panel_stack) - 1
        for i, (p, c) in enumerate(self._panel_stack):
            pw, ph = p.size_get()
            x = w - pw - self.stack_panel_offset * (n - i)
            p.geometry_set(x, 0, pw, h)

    def back(self):
        """Inform controller of a "back" action.

        This informs controller using C{self.callback_back}, that will get
        as parameter the reference to this window.
        """
        self.callback_back(self)

    def go_home(self):
        """Inform controller of a "go home" action.

        This informs controller using C{self.callback_go_home}, that will get
        as parameter the reference to this window.
        """
        self.callback_go_home(self)

    def toggle_fullscreen(self):
        """Inform controller of a "toggle_fullscreen" action.

        This informs controller using C{self.callback_toggle_fullscreen},
        that will get as parameter the reference to this window.
        """
        self.callback_toggle_fullscreen(self)

    def multitask_clicked(self):
        """Inform controller of a "multitask_clicked" action.

        This informs controller using C{self.callback_multitask_clicked},
        that will get as parameter the reference to this window.
        """
        self.callback_multitask_clicked(self)

    def handle_key_up(self, event):
        key_callback = self.key_up_bindings.get(event.keyname, None)
        if key_callback is not None:
            key_callback()

    def handle_key_down(self, event):
        key_callback = self.key_down_bindings.get(event.keyname, None)
        if key_callback is not None:
            key_callback()

    @edje.decorators.signal_callback("action,clicked", "back")
    def _cb_on_edje_clicked_back(self, *ignored):
        self.back()

    @edje.decorators.signal_callback("action,clicked", "go_home")
    def _cb_on_edje_clicked_go_home(self, *ignored):
        self.go_home()

    @edje.decorators.signal_callback("action,clicked", "options")
    def _cb_on_edje_clicked_options(self, *ignored):
        if self._current_screen:
            self._current_screen.options()

    @edje.decorators.signal_callback("action,clicked", "panel,back")
    def _cb_on_edje_clicked_panel_back(self, *ignored):
        if self.callback_panel_back:
            self.callback_panel_back()

    @edje.decorators.signal_callback("action,multitask,clicked",
                                     "titlebar:multitask")
    def _cb_on_edje_clicked_multitask(self, *ignored):
        self.multitask_clicked()

    def _reapply_visibility_settings(self):
        if self._titlebar_visible:
            self.signal_emit("show,titlebar", "")
        else:
            self.signal_emit("hide,titlebar", "")

        if self._btn_back_visible:
            self.back_button_show()
        else:
            self.back_button_hide()

        if self._btn_options_visible:
            self.options_button_show()
        else:
            self.options_button_hide()

    def multitask_state_set(self, state):
        if state < 0 or state > len(self.multitask_state_signals):
            raise ValueError("wrong state for multitask")

        ed = self.part_swallow_get("titlebar:multitask_icon")
        cmd = self.multitask_state_signals[state]
        ed.signal_emit("state,%s" % cmd, "")
        self._multitask_state = state

    def theme_changed(self, end_callback=None):
        exit_signal = "theme,exit"
        exit_signal_finished = exit_signal + ",finished"
        def cb(*ignored):
            self.signal_callback_del(exit_signal_finished, "", cb)

            base.EdjeWidget.theme_changed(self)
            self.multitask_state_set(self._multitask_state)

            self._btn_back.theme_changed()
            self._btn_options.theme_changed()

            for c in self._box_stat_icons.children_get():
                c.theme_changed()

            self._reapply_visibility_settings()

            # XXX: this re-swallow should be automatic, fix edje.
            self.part_swallow("titlebar:icons", self._box_stat_icons)

            if self._current_screen is not None:
                self.title_set(self._current_screen.title)
                self._current_screen.theme_changed()

            enter_signal = "theme,enter"
            if end_callback is not None:
                end_callback(self)
                enter_signal_finished = enter_signal + ",finished"
                def cb2(*ignored):
                    self.signal_callback_del(enter_signal_finished, "",cb2)
                    self.signal_emit("events,unblock", "")
                self.signal_callback_add(enter_signal_finished, "", cb2)
            self.signal_emit(enter_signal, "")

        self.signal_emit("events,block", "")
        self.signal_callback_add(exit_signal_finished, "", cb)
        self.signal_emit(exit_signal, "")

    def title_set(self, title):
        """Change window title.

        This doesn't do any animation, just set the view property and
        is meant for internal use only.
        """
        self.part_text_set("titlebar:title:cur_label", title)

    def title_replace(self, title):
        """Change window title using 'replace' animation.

        This will use the following Edje signals:
         - B{"title,transition,start,replace":} starts the animation.
         - B{"title,transition,reset":} reset to default position.
         - B{"title,transition,finished":} received when animation ends.

        @warning: this shouldn't be called by end-users, they should
            change screen's title that will take care of it.
        """
        ed = self.part_swallow_get("titlebar:title")

        def title_swap(*a):
            self.title_set(title)
            ed.signal_emit("title,transition,reset", "")
            ed.signal_callback_del("title,transition,finished", "", title_swap)
            self._anim_title = None

        if self._anim_title is not None:
            # prematurely finish the previous animation
            self._anim_title()

        self._anim_title = title_swap
        ed.part_text_set("new_label", title)
        ed.signal_callback_add("title,transition,finished", "", title_swap)
        ed.signal_emit("title,transition,start,replace", "")

    def _screen_set(self, scr):
        """Change window contents to the given screen.

        This doesn't do any animation, just set the view property and
        is meant for internal use only.

        @see: L{use()}
        """
        self.part_swallow("contents", scr)
        self._current_screen = scr
        self.title_set(scr.title)
        scr.focus = True

    def _screen_replace(self, scr):
        """Change window contents to the given screen using minor animations.

        This do minor animations, but not a transition.

        @see: L{transition_to()}
        """
        self.part_swallow("contents", scr)
        self._current_screen = scr
        scr.focus = True

    def use(self, scr):
        """Set the contents of the main window to be the given screen.

        If L{scr} provides a C{options()} method, then the "options" button
        will be enabled and when it's clicked it will call that method.

        @param scr: screen to use as window contents.
        @type scr: L{screen.Screen}

        @see: L{transition_to()}
        """
        if scr is self._current_screen:
            return

        self._custom_controls_check(self._current_screen, scr, False)

        if self._current_screen:
            self._screen_replace(scr)
        else:
            self._screen_set(scr)

    def transition_to(self, sub_view_new, end_callback=None):
        """Change screens using a transition animation.

        Transitions from current screen, also referred as C{sub_view_old},
        to the new screen C{sub_view_new}. It will call:
         - L{sub_view_old.transition_to(sub_view_new)
             <screen.Screen.transition_to()>}
         - L{sub_view_new.transition_from(sub_view_old)
             <screen.Screen.transition_from()>}

        @param sub_view_new: screen to use as window contents.
        @type sub_view_new: L{screen.Screen}
        @param end_callback: function to call when transition finishes.
        @type end_callback: function(window, sub_view_old, sub_view_new)
        """
        sub_view_old = self._current_screen
        self.part_swallow("new_contents", sub_view_new)
        self.signal_emit("events,block", "")

        def inform_end(*ignored):
            self.part_unswallow(sub_view_new)
            if end_callback is not None:
                end_callback(self, sub_view_old, sub_view_new)
            self._screen_replace(sub_view_new)
            sub_view_old.transition_finished(sub_view_old, sub_view_new)
            sub_view_new.transition_finished(sub_view_old, sub_view_new)
            self.signal_emit("events,unblock", "")

        self.title_replace(sub_view_new.title)
        if self._current_screen is None:
            inform_end()
        else:
            cb = generate_multiple_wait(2, inform_end)
            self._custom_controls_check(sub_view_old, sub_view_new)
            sub_view_old.transition_to(sub_view_new, cb)
            sub_view_new.transition_from(sub_view_old, cb)

    def _custom_controls_check(self, current_screen, new_screen, show_controls=True):
        def hide_back_finished(*ignored):
            bt = new_screen.custom_back_button()
            if bt:
                self.back_button_set(bt)
            else:
                self.back_button_reset()

            if show_controls:
                self.back_button_show()

        def hide_options_finished(*ignored):
            bt = new_screen.custom_options_button()
            if bt:
                self.options_button_set(bt)
            else:
                self.options_button_reset()

            if show_controls and new_screen.has_options():
                self.options_button_show()

        if current_screen:
            if current_screen.custom_back_button() or new_screen.custom_back_button():
                self.back_button_hide(hide_back_finished)

            if current_screen.custom_options_button() or new_screen.custom_options_button():
                self.options_button_hide(hide_options_finished)
            elif new_screen.has_options():
                show_controls and self.options_button_show()
            else:
                self.options_button_hide()
        else:
            hide_back_finished(self)
            hide_options_finished(self)

    def _emit_and_inform_finish(self, signal, source, end_callback=None):
        if end_callback is not None:
            signal_finished = signal + ",finished"
            def cb(*ignored):
                self.signal_callback_del(signal_finished, source, cb)
                end_callback(self)
            self.signal_callback_add(signal_finished, source, cb)
        self.signal_emit(signal, source)

    def titlebar_hide(self, end_callback=None):
        """Hide title bar using animation.

        @param end_callback: function to callback when animation finishes.
        @type end_callback: function(window)
        """
        self._titlebar_visible = False
        self._emit_and_inform_finish("transition,hide,titlebar", "",
                                     end_callback)

    def titlebar_show(self, end_callback=None):
        """Show title bar using animation.

        @param end_callback: function to callback when animation finishes.
        @type end_callback: function(window)
        """
        self._titlebar_visible = True
        self._emit_and_inform_finish("transition,show,titlebar", "",
                                     end_callback)

    def controls_hide(self, transition=True, end_callback=None):
        """Hide controls, animating accordingly to transition param

        @param transition: whether hide with animation or not
        @param end_callback: function to callback when animation finishes.
        @type end_callback: function(window)
        """
        self._btn_back_visible = self._btn_options_visible = False
        signal = transition and "transition,hide,controls" or "hide,controls"
        self._emit_and_inform_finish(signal, "", end_callback)

    def back_button_hide(self, end_callback=None):
        """Hide back button using animation

        @param end_callback: function to callback when animation finishes.
        @type end_callback: function(window)
        """
        self._btn_back_visible = False
        self._emit_and_inform_finish("transition,hide,back", "",
                                     end_callback)

    def options_button_hide(self, end_callback=None):
        """Hide options button using animation

        @param end_callback: function to callback when animation finishes.
        @type end_callback: function(window)
        """
        self._btn_options_visible = False
        self._emit_and_inform_finish("transition,hide,options", "",
                                     end_callback)

    def controls_show(self, end_callback=None):
        """Show controls using animation.

        @param end_callback: function to callback when animation finishes.
        @type end_callback: function(window)
        """
        self.back_button_show(end_callback)
        if self._current_screen.has_options():
            self.options_button_show(end_callback)

    def back_button_show(self, end_callback=None):
        """Show back button using animation

        @param end_callback: function to callback when animation finishes.
        @type end_callback: function(window)
        """
        self._btn_back_visible = True
        self._emit_and_inform_finish("transition,show,back", "",
                                      end_callback)

    def options_button_show(self, end_callback=None):
        """Show options button using animation

        @param end_callback: function to callback when animation finishes.
        @type end_callback: function(window)
        """
        self._btn_options_visible = True
        self._emit_and_inform_finish("transition,show,options", "",
                                      end_callback)

    def control_buttons_set(self, btn_back, btn_options):
        """Show custom control buttons without animation.
        """
        self.back_button_set(btn_back)
        self.options_button_set(btn_options)

    def back_button_set(self, btn_back):
        """Set custom back button without animation.

        @param btn_back: the custom image to be set
        @type btn_back: edje with same group of bt_back_default
                        in default_main.edc
        """
        self._swallow_replace("bt_back", self._cur_btn_back, btn_back)
        self._cur_btn_back = btn_back

    def options_button_set(self, btn_options):
        """Set custom options button without animation.

        @param btn_options: the custom image to be set
        @type btn_options: edje with same group of bt_options_default
                           in default_main.edc
        """
        self._swallow_replace("bt_options", self._cur_btn_options, btn_options)
        self._cur_btn_options = btn_options

    def control_buttons_reset(self):
        """Restore default control buttons without animation.
        """
        self.back_button_reset()
        self.options_button_reset()

    def back_button_reset(self):
        """Restore default control back button without animation.
        """
        self.back_button_set(self._btn_back)

    def options_button_reset(self):
        """Restore default control options button without animation.
        """
        self.options_button_set(self._btn_options)

    def add_status_icon(self, obj):
        """Add StatusIcon to main window.

        @warning: this function is not meant to end users, but to be used
                  by StatusIcon internally. You should just instantiate
                  StatusIcon()  class and it would add automatically.
        """
        self._box_stat_icons.append(obj)
        w, h = self._box_stat_icons.required_size_get()
        self._box_stat_icons.resize(w, h)
        edje.extern_object_min_size_set(self._box_stat_icons, w, h)
        edje.extern_object_max_size_set(self._box_stat_icons, w, h)

    def del_status_icon(self, obj):
        """Removes StatusIcon from main window.

        @warning: this function is not meant to end users, but to be used
                  by StatusIcon internally.
        """
        self._box_stat_icons.remove(obj)
        w, h = self._box_stat_icons.required_size_get()
        self._box_stat_icons.resize(w, h)
        edje.extern_object_min_size_set(self._box_stat_icons, w, h)
        edje.extern_object_max_size_set(self._box_stat_icons, w, h)

    def StatusIcon(self, icon, theme=None):
        """Factory of StatusIcon associated with this window.

        @rtype: L{StatusIcon}
        """
        return StatusIcon(self, icon, theme)

    def show_notify(self, notify):
        assert self._notify is None

        notify.signal_emit("notify,block_area,show", "")
        self.part_swallow("notify_contents", notify)
        notify.show()
        self._notify = notify

    def hide_notify(self, notify):
        assert notify == self._notify

        notify = self._notify
        notify.hide()
        self.part_unswallow(notify)
        notify.signal_emit("notify,block_area,hide", "")
        self._notify = None

    def NotifyWindow(self, group, message, theme=None):
        return NotifyWindow(self, group, message, theme)

    def _panel_stack_animation(self, panel, end_callback):
        if not self._panel_stack:
            def inform_end(*ignored):
                if end_callback:
                    end_callback(self, panel)
            cb = generate_multiple_wait(2, inform_end)

            def cb_edje(*ignored):
                self.signal_callback_del("panel,bg,show,finished", "", cb_edje)
                cb()

            self.signal_callback_add("panel,bg,show,finished", "", cb_edje)
            self.signal_emit("panel,bg,show", "")
        else:
            self._panel_stack[-1][0].signal_emit("panel,block_area,show", "")
            self._panel_stack[-1][0].focus = False
            cb = end_callback

        w, h = self.size
        pw, ph = panel.size
        orig = tuple((p, c, p.pos[0], c.color[0]) for p, c in self._panel_stack)

        def anim(value, orig):
            x, p = value
            panel.move(int(x), 0)

            for pan, clip, px, color in orig:
                pan.move(int(px - self.stack_panel_offset * p), 0)
                color = max(55, int(color - 50.0 * p))
                clip.color_set(color, color, color, 255)

            if p >= 1.0 and cb:
                panel.signal_emit("panel,block_area,hide", "")
                cb(self, panel)
                panel.focus = True

        panel.signal_emit("panel,block_area,show", "")
        panel.geometry_set(w, 0, pw, h)
        panel.show()
        start = (w, 0.0)
        end = (w - pw, 1.0)
        TimelineAnimation(start, end, self.stack_panel_timeout, anim, orig)

    def panel_stack(self, panel, end_callback=None):
        """Stack the given panel in the panel area.

        This will run an animation to stack the panel, after it is finished
        end_callback will be called.

        Signature: C{function(window, panel)}
        """
        self._panel_stack_animation(panel, end_callback)

        clip = self.evas.Rectangle(color=(255, 255, 255, 255),
                                   geometry=(-9999, -9999, 99999, 99999))
        clip.show()
        panel.clip_set(clip)
        self._panel_stack.append((panel, clip))

    def _panel_unstack_animation(self, panel, end_callback):
        if not self._panel_stack:
            def inform_end(*ignored):
                if end_callback:
                    end_callback(self, panel)
            cb = generate_multiple_wait(2, inform_end)

            def cb_edje(*ignored):
                self.signal_callback_del("panel,bg,hide,finished", "", cb_edje)
                cb()

            self.signal_callback_add("panel,bg,hide,finished", "", cb_edje)
            self.signal_emit("panel,bg,hide", "")
        else:
            cb = end_callback

        w, h = self.size
        pw, ph = panel.size
        orig = tuple((p, c, p.pos[0]) for (p, c) in self._panel_stack)

        def anim(value, orig):
            x, p = value
            panel.move(int(x), 0)

            last = len(self._panel_stack) - p
            for i, (pan, clip, px) in enumerate(orig):
                pan.move(int(px + self.stack_panel_offset * p), 0)
                color = min(255, max(55, int(255 - 50 * (last - i))))
                clip.color_set(color, color, color, 255)

            if p >= 1.0:
                panel.hide()
                if self._panel_stack:
                    self._panel_stack[-1][0].signal_emit(
                        "panel,block_area,hide", "")
                    self._panel_stack[-1][0].focus = True
                else:
                    # Get the focus back from Settings
                    self._current_screen.focus = True
                if cb:
                    cb(self, panel)

        panel.focus = False
        panel.signal_emit("panel,block_area,show", "")
        start = (w - pw, 0.0)
        end = (w, 1.0)
        TimelineAnimation(start, end, self.stack_panel_timeout, anim, orig)

    def panel_unstack(self, end_callback=None):
        """Unstack (pop) the last panel from the panel area.

        This will run an animation to unstack the panel, after it is finished
        end_callback will be called.

        Signature: C{function(window, panel)}
        """
        if not self._panel_stack:
            if end_callback:
                end_callback(self, None)
            return

        panel, clip = self._panel_stack.pop()
        clip.delete()
        self._panel_unstack_animation(panel, end_callback)

    def _swallow_replace(self, name, old_part, new_part):
        if old_part == new_part:
            return

        if old_part:
            self.part_unswallow(old_part)
            old_part.hide()

        new_part.show()
        self.part_swallow(name, new_part)

    @evas.decorators.del_callback
    def _cb_on_delete(self):
        self._btn_back.delete()
        self._btn_options.delete()
