# ubuntuone.syncdaemon.main - main SyncDaemon innards
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#         John Lenton <john.lenton@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" SyncDaemon Main"""
import dbus
from dbus.mainloop.glib import DBusGMainLoop
import gnomekeyring
import logging
import os
import sys

from twisted.internet import defer, reactor, task

from oauth import oauth
from ubuntuone.syncdaemon import (
    action_queue,
    config,
    dbus_interface,
    event_queue,
    filesystem_manager,
    hash_queue,
    events_nanny,
    local_rescan,
    states,
    sync,
    volume_manager,
)
from ubuntuone import syncdaemon
from ubuntuone.syncdaemon.state import SyncDaemonStateManager


class WaitingHelpingHandler(object):
    """
    An auxiliary class that helps wait for events
    """
    def __init__(self, event_queue, waiting_events, waiting_kwargs,
                 result=None):
        self.deferred = defer.Deferred()
        self.event_queue = event_queue
        self.result = result
        self.waiting_events = waiting_events
        self.waiting_kwargs = waiting_kwargs
        event_queue.subscribe(self)

    def handle_default(self, event, *args, **kwargs):
        """Got an event: fire if it's one we want"""
        if event in self.waiting_events:
            for wk, wv in self.waiting_kwargs.items():
                if not (wk in kwargs and kwargs[wk] == wv):
                    return
            self.fire()

    def fire(self):
        """start fire the callback"""
        self.event_queue.unsubscribe(self)
        reactor.callLater(0, lambda: self.deferred.callback(self.result))


class Main(object):
    """ The one who executes the syncdaemon """

    def __init__(self, root_dir, shares_dir, data_dir, partials_dir,
                 host='fs-1.one.ubuntu.com', port=443, dns_srv=None, ssl=True,
                 disable_ssl_verify=False,
                 realm='https://ubuntuone.com', glib_loop=False,
                 mark_interval=120, dbus_events=False,
                 handshake_timeout=30, max_handshake_timeouts=10,
                 shares_symlink_name='Shared With Me',
                 read_limit=None, write_limit=None, throttling_enabled=None):
        """ create the instance. """
        self.root_dir = root_dir
        self.shares_dir = shares_dir
        self.shares_dir_link = os.path.join(self.root_dir, shares_symlink_name)
        self.data_dir = data_dir
        self.partials_dir = partials_dir
        self.logger = logging.getLogger('ubuntuone.SyncDaemon.Main')
        self.host = host
        self.port = port
        self.dns_srv = dns_srv
        self.ssl = ssl
        self.disable_ssl_verify = disable_ssl_verify
        self.realm = realm
        self.token = None
        user_config = config.get_user_config()
        if read_limit is None:
            read_limit = user_config.get_throttling_read_limit()
        if write_limit is None:
            write_limit = user_config.get_throttling_write_limit()
        if throttling_enabled is None:
            throttling_enabled = user_config.get_throttling()

        self.vm = volume_manager.VolumeManager(self)
        self.fs = filesystem_manager.FileSystemManager(data_dir,
                                                       self.partials_dir,
                                                       self.vm)
        self.event_q = event_queue.EventQueue(self.fs)
        self.fs.register_eq(self.event_q)
        self.oauth_client = OAuthClient(self.realm)
        self.state = SyncDaemonStateManager(self, handshake_timeout,
                                            max_handshake_timeouts)
        # subscribe VM to EQ
        self.event_q.subscribe(self.vm)
        self.vm.init_root()
        # we don't have the oauth tokens yet, we 'll get them later
        self.action_q = action_queue.ActionQueue(self.event_q, host, port,
                                                 self.dns_srv, ssl,
                                                 disable_ssl_verify,
                                                 read_limit, write_limit,
                                                 throttling_enabled)
        self.hash_q = hash_queue.HashQueue(self.event_q)
        events_nanny.DownloadFinishedNanny(self.fs, self.event_q, self.hash_q)

        self.sync = sync.Sync(self)
        self.lr = local_rescan.LocalRescan(self.vm, self.fs,
                                           self.event_q, self.action_q)

        if not glib_loop:
            self.bus = dbus.SessionBus()
        else:
            loop = DBusGMainLoop(set_as_default=True)
            self.bus = dbus.SessionBus(loop)
        self.dbus_iface = dbus_interface.DBusInterface(self.bus, self,
                                                       send_events=dbus_events)
        self.action_q.content_queue.set_change_notification_cb(
            self.dbus_iface.status.emit_content_queue_changed)
        self.logger.info("Using %s as root dir", self.root_dir)
        self.logger.info("Using %s as data dir", self.data_dir)
        self.logger.info("Using %s as shares root dir", self.shares_dir)
        self.mark = task.LoopingCall(self.log_mark)
        self.mark.start(mark_interval)

    def log_mark(self):
        """ log a "mark" that includes the current AQ state and queue size"""
        self.logger.note("%s %s (state: %s; queues: metadata: %d; content: %d;"
                         " hash: %d, fsm-cache: hit=%d miss=%d) %s" % ('-'*4,
                                         'MARK', self.state.name,
                                         len(self.action_q.meta_queue),
                                         len(self.action_q.content_queue),
                                         len(self.hash_q),
                                         self.fs.fs.cache_hits,
                                         self.fs.fs.cache_misses, '-'*4))

    def wait_for_nirvana(self, last_event_interval=0.5):
        """Get a deferred that will fire when there are no more
        events or transfers."""
        self.logger.debug('wait_for_nirvana(%s)' % last_event_interval)
        d = defer.Deferred()
        def start():
            """request the event empty notification"""
            self.logger.debug('starting wait_for_nirvana')
            self.event_q.add_empty_event_queue_callback(callback)
        def callback():
            """event queue is empty"""
            if not (self.state == states.IDLE
                    and not self.action_q.meta_queue
                    and not self.action_q.content_queue
                    and self.hash_q.empty()):
                self.logger.debug("I can't attain Nirvana yet."
                                  " [state: %s; queues:"
                                  " metadata: %d; content: %d; hash: %d]"
                                  % (self.state.name,
                                     len(self.action_q.meta_queue),
                                     len(self.action_q.content_queue),
                                     len(self.hash_q)))
                return
            self.logger.debug("Nirvana reached!! I'm a Buddha")
            self.event_q.remove_empty_event_queue_callback(callback)
            d.callback(True)
        reactor.callLater(last_event_interval, start)
        return d

    def wait_for(self, *waiting_events, **waiting_kwargs):
        """defer until event appears"""
        return WaitingHelpingHandler(self.event_q,
                                     waiting_events,
                                     waiting_kwargs).deferred

    def start(self):
        """setup the daemon to be ready to run"""
        # hook the event queue to the root dir
        self.event_q.push('SYS_WAIT_FOR_LOCAL_RESCAN')
        self.event_q.inotify_add_watch(self.root_dir)

        # do the local rescan
        self.logger.note("Local rescan starting...")
        d = self.lr.start()

        def _wait_for_hashq():
            """
            Keep on calling this until the hash_q finishes.
            """
            if len(self.hash_q):
                self.logger.info("hash queue pending. Waiting for it...")
                reactor.callLater(.1, _wait_for_hashq)
            else:
                self.logger.info("hash queue empty. We are ready!")
                # nudge the action queue into action
                self.event_q.push('SYS_LOCAL_RESCAN_DONE')

        def local_rescan_done(_):
            '''After local rescan finished.'''
            self.logger.note("Local rescan finished!")
            _wait_for_hashq()

        def stop_the_press(failure):
            '''Something went wrong in LR, can't continue.'''
            self.logger.error("Local rescan finished with error: %s",
                                                failure.getBriefTraceback())
            self.event_q.push('SYS_UNKNOWN_ERROR')

        d.addCallbacks(local_rescan_done, stop_the_press)
        return d

    def shutdown(self, with_restart=False):
        """ shutdown the daemon; optionally restart it """
        self.event_q.push('SYS_DISCONNECT')
        self.event_q.shutdown()
        self.hash_q.shutdown()
        self.dbus_iface.shutdown(with_restart)
        self.mark.stop()

    def restart(self):
        """
        Restart the daemon.

        This ultimately shuts down the daemon and asks dbus to start one again.
        """
        self.quit(exit_value=42, with_restart=True)

    def get_root(self, root_mdid):
        """
        Ask que AQ for our root's uuid
        """
        def _worker():
            """
            Actually do the asking
            """
            d = self.action_q.get_root(root_mdid)
            def root_node_cb(root):
                """ root node fetched callback. """
                root_mdid = self.vm.on_server_root(root)
                self.action_q.uuid_map.set(root_mdid, root)
            d.addCallback(root_node_cb)
            return d
        if self.action_q.client is None:
            # aq not yet connected
            self.action_q.deferred.addCallback(_worker)
            return self.action_q.deferred
        else:
            return _worker()

    def check_version(self):
        """
        Check the client protocol version matches that of the server.
        """
        self.action_q.check_version()

    def authenticate(self):
        """
        Do the OAuth dance.
        """
        self.action_q.authenticate(self.oauth_client.consumer)

    def set_capabilities(self):
        """Set the capabilities with the server"""
        self.logger.debug("capabilities query: %r", syncdaemon.REQUIRED_CAPS)
        self.action_q.set_capabilities(syncdaemon.REQUIRED_CAPS)

    def server_rescan(self):
        """
        Do the server rescan
        """
        self.action_q.server_rescan(self.fs.get_data_for_server_rescan)

    def set_oauth_token(self, key, secret):
        """ Sets the oauth token """
        self.token = oauth.OAuthToken(key, secret)

    def get_access_token(self):
        """Return the access token or a new one"""
        if self.token:
            return self.token
        else:
            return self.oauth_client.get_access_token()

    def get_rootdir(self):
        """ Returns the base dir/mount point"""
        return self.root_dir

    def quit(self, exit_value=0, with_restart=False):
        """ shutdown and stop the reactor. """
        self.shutdown(with_restart)
        if reactor.running:
            reactor.stop()
        else:
            sys.exit(exit_value)


class NoAccessToken(Exception):
    """No access token available."""


class OAuthClient(object):
    """ Basic OAuth client, just grab the token from the keyring. """

    def __init__(self, realm, consumer_key='ubuntuone',
                 consumer_secret='hammertime'):
        """ create the instance and setup the modules """
        self.realm = realm
        self.consumer = oauth.OAuthConsumer(consumer_key, consumer_secret)

    def get_access_token(self):
        """Get the access token from the keyring.

        If no token is available in the keyring, `NoAccessToken` is raised.
        """
        try:
            items = gnomekeyring.find_items_sync(
                gnomekeyring.ITEM_GENERIC_SECRET,
                {'ubuntuone-realm': self.realm,
                 'oauth-consumer-key': self.consumer.key})
            return oauth.OAuthToken.from_string(items[0].secret)
        except gnomekeyring.NoMatchError:
            raise NoAccessToken("No access token found.")
        except (gnomekeyring.NoKeyringDaemonError,
                gnomekeyring.DeniedError), e:
            raise NoAccessToken("Error while getting the token: %s" % type(e))


