# ubuntuone.syncdaemon.fsm.tests.test_fsm 
#
# Author: Lucio Torre <lucio.torre@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
tests for fsm that depend on python uno
"""
import unittest
import os


from ubuntuone.syncdaemon.fsm import fsm

def p(name):
    """make a full path from here."""
    return os.path.join(os.path.dirname(__file__), name)

class TestParse(unittest.TestCase):
    'Test fsm validation'
    
    def test_one_event(self):
        'test parsing a simple machine'
        f = fsm.StateMachine(p("test_one_event.ods"))
        f.validate()
        
    def test_two_events(self):
        'test parsing a machine with two events'
        f = fsm.StateMachine(p("test_two_events.ods"))
        f.validate()

    def test_bang(self):
        'test not event expansion'
        f = fsm.StateMachine(p("test_bang.ods"))
        f.validate()
    
    def test_transition_twice(self):
        'test error on duplicate transition'
        f = fsm.StateMachine(p("test_transition_twice.ods"))
        self.assertRaises(fsm.ValidationFailed, f.validate)
        self.assertEquals(len(f.errors), 1)
        
    def test_missing_source_state(self):
        'test incomplete state transition coverage'
        f = fsm.StateMachine(p("test_missing_source_state.ods"))
        self.assertRaises(fsm.ValidationFailed, f.validate)
        self.assertEquals(len(f.errors), 1)
    
    def test_missing_param_values(self):
        'test incomplete param transition coverage'
        f = fsm.StateMachine(p("test_missing_param_values.ods"))
        self.assertRaises(fsm.ValidationFailed, f.validate)
        self.assertEquals(len(f.errors), 4)
    
        
    def test_two_missing_source_state(self):
        'test incomplete state transition coverage'
        f = fsm.StateMachine(p("test_two_missing_source_state.ods"))
        self.assertRaises(fsm.ValidationFailed, f.validate)
        self.assertEquals(len(f.errors), 2)

    def test_star_event(self):
        'test expansion of one star in event columns'
        f = fsm.StateMachine(p("test_star_event.ods"))
        f.validate()
    
    def test_two_star_event(self):
        'test expansion of two stars in event columns'
        f = fsm.StateMachine(p("test_two_star_event.ods"))
        f.validate()
    
    def test_star_param(self):
        'test expansion of one star in param columns'
        f = fsm.StateMachine(p("test_star_param.ods"))
        f.validate()
    
    def test_two_star_param(self):
        'test expansion of two stars in param columns'
        f = fsm.StateMachine(p("test_two_star_param.ods"))
        f.validate()
    
    def test_invalid(self):
        'test expansion of two stars in param columns'
        f = fsm.StateMachine(p("test_invalid.ods"))
        f.validate()

    def test_invalid_expand(self):
        'test expansion of two stars in param columns'
        f = fsm.StateMachine(p("test_invalid_expand.ods"))
        f.validate()
        
        
    def test_star_event_repeat(self):
        'test expansion of stars that cover too much'
        f = fsm.StateMachine(p("test_star_event_repeat.ods"))
        self.assertRaises(fsm.ValidationFailed, f.validate)
        self.assertEquals(len(f.errors), 1)

    def test_out_equal(self):
        'test expansion of "=" in state out'
        f = fsm.StateMachine(p("test_out_equal.ods"))
        f.validate()
        for s in f.states.values():
            for t in s.transitions.values():
                for k in t.source:
                    self.assertEquals(t.source[k], t.target[k])

    def test_out_equal_star(self):
        'test expansion of "=" in state out'
        f = fsm.StateMachine(p("test_out_equal_star.ods"))
        f.validate()
        for s in f.states.values():
            for t in s.transitions.values():
                for k in t.source:
                    self.assertEquals(t.source[k], t.target[k],
                        "on transition %s target is %s"%(t, t.target))
                    
    def test_equal_wrong_places(self):
        'make sure "=" are not allowed on state or params'
        f = fsm.StateMachine(p("test_equal_wrong_place.ods"))
        self.assertRaises(fsm.ValidationFailed, f.validate)
        # this should be two errors
        # but more errors happen as there is no clear interpretation of
        # the table in this case
        self.assertEquals(len(f.errors), 5)
        
    def test_param_na(self):
        'test that na param columns are ignored'
        f = fsm.StateMachine(p("test_param_na.ods"))
        f.validate()
        self.assertEqual(f.events["EVENT_2"].transitions[0].parameters.keys(),
                         [u"MV2"],)

    def test_func_na(self):
        'test that na param columns are ignored'
        f = fsm.StateMachine(p("test_func_na.ods"))
        f.validate()
        # the state
        s = f.states[fsm.hash_dict(dict(SV1="T"))]
        # the transition
        t = "EVENT_1", fsm.hash_dict(dict(MV1="T", MV2="T"))
        self.assertFalse(t in s.transitions)

    
    
def test_suite():
    # pylint: disable-msg=C0111
    if "HAS_OOFFICE" in os.environ:
        return unittest.TestLoader().loadTestsFromName(__name__)
    else:
        return unittest.TestSuite()
        
        
if __name__ == "__main__":
    unittest.main()
